*     ------------------------------------------------------------------
*     File t3qp.f  (Unix version)
*     Illustrates using SNOPT on a quadratic program.
*
*     15 May 1998: First   version.
*     27 Feb 1998: Current version.
*     ------------------------------------------------------------------
      program            t3main
      implicit           double precision (a-h,o-z)

      parameter        ( maxm   = 1000,
     $                   maxn   = 1000,
     $                   maxne  = 3000,
     $                   nName  = 1 )

      character*8        Prob
      character*8        Names(nName)
      integer*4          ha(maxne) , hs(maxn+maxm)
      integer            ka(maxn+1)
      double precision   a(maxne)  , bl(maxn+maxm), bu(maxn+maxm),
     $                   x(maxn+maxm), pi(maxm)   , rc(maxn+maxm)

*     SNOPT workspace

      parameter          (  lenrw = 20000)
      double precision   rw(lenrw)
      parameter          (  leniw = 10000) 
      integer            iw(leniw)
      parameter          (  lencw =   500) 
      character*8        cw(lencw)

      logical            byname
      character*20       lfile
      external           t3obj, dumcon

*     ------------------------------------------------------------------
*     Specify some of the SNOPT files.
*     iSpecs  is the Specs file   (0 if none).
*     iPrint  is the Print file   (0 if none).
*     iSumm   is the Summary file (0 if none).
*
*     nout    is an output file used here by t3qp.

      iSpecs = 4
      iPrint = 15
      iSumm  = 6
      nout   = 6

      byname = .true.
 
      if ( byname ) then

*        Unix and DOS systems.  Open the Specs and print files.

         lfile = 't3qp.spc'
         open( iSpecs, file=lfile, status='OLD',     err=900 )

         lfile = 't3qp.out'
         open( iPrint, file=lfile, status='UNKNOWN', err=900 )
      end if

*     ------------------------------------------------------------------
*     First,  snInit MUST be called to initialize optional parameters 
*     to their default values.
*     ------------------------------------------------------------------
      call snInit( iPrint, iSumm,
     $             cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Read a Specs file (Optional).
*     ------------------------------------------------------------------
      call snSpec( iSpecs, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )

      if (inform .ge. 2) then
         write(nout, *) 'iSpecs > 0 but no Specs file found'
         go to 800
      end if

*     Set up the data structure for the sparse Jacobian.
*     Assign dummy values for the nonlinear elements. 

      call t3data( maxm, maxn, maxne, inform, 
     $             m, n, ne, nnCon, nnObj, nnJac,
     $             iObj, ObjAdd, Prob,
     $             a, ha, ka, bl, bu, hs, x, pi )

*     ------------------------------------------------------------------
*     Specify any options not set in the Specs file.
*     i1 and i2 may refer to the Print and Summary file respectively.
*     Setting them to 0 suppresses printing.
*     ------------------------------------------------------------------
      itnlim = 250
      i1     =   0
      i2     =   0
      call snseti( 'Iterations        ', itnlim, i1, i2, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Go for it, using a Cold start.
*     hs     need not be set if a basis file is to be input.
*            Otherwise, each hs(1:n) should be 0, 1, 2, 3, 4, or 5.
*            The values are used by the Crash procedure m2crsh
*            to choose an initial basis B.
*            If hs(j) = 0 or 1, column j is eligible for B.
*            If hs(j) = 2, column j is initially superbasic (not in B).
*            If hs(j) = 3, column j is eligible for B and is given
*                          preference over columns with hs(j) = 0 or 1.
*            If hs(j) = 4 or 5, column j is initially nonbasic.
*     ------------------------------------------------------------------
      call snopt ( 'Cold', m, n, ne, nName,
     $             nnCon, nnObj, nnJac,
     $             iObj, ObjAdd, Prob,
     $             dumcon, t3obj,
     $             a, ha, ka, bl, bu, Names,
     $             hs, x, pi, rc, 
     $             inform, mincw, miniw, minrw,
     $             nS, nInf, sInf, Obj,
     $             cw, lencw, iw, leniw, rw, lenrw,
     $             cw, lencw, iw, leniw, rw, lenrw )

      write(nout, *) ' '

      if (inform .eq. 42 .or. inform .eq. 43 .or. inform .eq. 44) then
         write(nout, *) 'Estimate of required lencw:', mincw
         write(nout, *) 'Estimate of required leniw:', miniw
         write(nout, *) 'Estimate of required lenrw:', minrw
      else
         write(nout, *) 'snopt finished.'
         write(nout, *) 'inform =', inform
         write(nout, *) 'nInf   =', nInf
         write(nout, *) 'sInf   =', sInf
         write(nout, *) 'Obj    =', Obj
      end if

      if (inform .ge. 20) then
         write(nout, *) ' '
         write(nout, *) 'STOPPING because of error condition'
      end if

  800 if ( byname ) then
         close(iSpecs)
         close(iPrint)
      end if

      stop

*     ------------------------------------------------------------------
*     File Error.
*     ------------------------------------------------------------------
  900 write(nout, 4000) 'Error while opening file', lfile
      stop

 4000 format(/  a, 2x, a  )

*     end of t3main.
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine t3obj ( mode, n, x, f, g, nState,
     $                   cu, lencu, iu, leniu, ru, lenru )

      implicit           double precision (a-h,o-z)
      double precision   x(n), g(n)

      character*8        cu(lencu)
      integer            iu(leniu)
      double precision   ru(lenru)

*     ------------------------------------------------------------------
*     This is funobj for problem t3qp.
*     f(x) = 1/2 x'Qx,   g(x) = Qx.
*     ------------------------------------------------------------------
      parameter        ( zero = 0.0d+0,  half = 0.5d+0,
     $                   two  = 2.0d+0,  four = 4.0d+0,
     $                   maxn = 10 )
      double precision   Q(maxn,maxn)
      save               Q

      if (nState .eq. 1) then

*        Define Q on the first entry.
*        Here we assume n = 3.

         Q(1,1) = four
         Q(1,2) = two
         Q(1,3) = two
         Q(2,2) = four
         Q(2,3) = zero
         Q(3,3) = two

*        Make Q symmetric.

         do 20, j = 1, n-1
            do 10, i = 2, n
               Q(i,j) = Q(j,i)
   10       continue
   20    continue
      end if

*     Compute f and g on all entries.
*     We first compute g = Qx, then f = 1/2 x'g.
*     In Fortran it is best to run down the columns of Q.

      do 200, i = 1, n      
         g(i)   = zero
  200 continue

      do 300, j = 1, n
         xj     = x(j)
         do 250, i = 1, n
            g(i)  = g(i)  +  Q(i,j) * xj
  250    continue
  300 continue

      f      = half  *  ddot  ( n, x, 1, g, 1 )

*     end of t3obj  (funobj for t3qp)
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine dumcon( mode, nnCon, nnJac, neJac, x, fCon, gCon,
     $                   nState, cu, lencu, iu, leniu, ru, lenru )

      implicit           double precision(a-h,o-z)
      double precision   x(nnJac), fCon(nnCon), gCon(neJac)

      character*8        cu(lencu)
      integer            iu(leniu)
      double precision   ru(lenru)

*     ==================================================================
*     Problem t3qp.
*     No nonlinear constraints.
*     ==================================================================

*     Relax

*     end of dummy constraints for t3qp
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine t3data( maxm, maxn, maxne, inform, 
     $                   m, n, ne, nnCon, nnObj, nnJac,
     $                   iObj, ObjAdd, Prob,
     $                   a, ha, ka, bl, bu, hs, x, pi )

      implicit           double precision (a-h,o-z)
      character*8        Prob
      integer            ha(maxne)   , hs(maxn+maxm)
      integer            ka(maxn+1)
      double precision   a(maxne)    , bl(maxn+maxm), bu(maxn+maxm),
     $                   x(maxn+maxm), pi(maxm)

*     ------------------------------------------------------------------
*     Define the problem.
*     (1) Compute l, u, and A so that the constraints are ranges of the
*         form  l <= Ax <= u.
*         Store l and u in bl(n+1:n+m) and bu(n+1:n+m).
*
*     (2) Set up the constants ObjAdd and iObj so that the explicit
*         objective is 
*             ObjAdd + (row iObj of A)'*x + f(x) 
*
*     On entry,
*     maxm, maxn, maxne are upper limits on m, n, ne.
*
*     On exit,
*     inform  is 0 if there is enough storage, 1 otherwise.
*     m       is the number of nonlinear and linear constraints.
*     n       is the number of variables.
*     ne      is the number of nonzeros in a(*).
*     nnCon   is the number of nonlinear constraints (they come first).
*     nnObj   is the number of nonlinear objective variables.
*     nnJac   is the number of nonlinear Jacobian variables.
*     a       is the constraint matrix (Jacobian), stored column-wise.
*     ha      is the list of row indices for each nonzero in a(*).
*     ka      is a set of pointers to the beginning of each column of a.
*     bl      is the lower bounds on x and s.
*     bu      is the upper bounds on x and s.
*     hs(1:n) is a set of initial states for each x  (0,1,2,3,4,5).
*     x (1:n) is a set of initial values for x.
*     pi(1:m) is a set of initial values for the dual variables pi.
*     ------------------------------------------------------------------
      parameter         (bplus   =  1.0d+21)

*     Name the Problem.

      Prob = 't3qp....'

      n     =  3
      m     =  2
      ne    =  6               ! n*m for a dense A.

*     Check if there is enough storage.

      inform = 0
      if (m      .gt. maxm ) inform = 1
      if (n      .gt. maxn ) inform = 1
      if (ne     .gt. maxne) inform = 1
      if (inform .gt.   0  ) return

      nnCon   = 0
      nnJac   = 0
      nnObj   = n

*     The linear objective is row 2 of a.

      iObj    = 2
      ObjAdd  = 0.0d+0

*     Column 1.

      ka( 1) =  1

      ha( 1) =  1
      ha( 2) =  2 

      a ( 1) =  1.0d+0
      a ( 2) = -8.0d+0

*     Column 2.

      ka( 2) =  3

      ha( 3) =  1
      ha( 4) =  2

      a ( 3) =  1.0d+0
      a ( 4) = -6.0d+0

*     Column 3.

      ka( 3) =  5

      ha( 5) =  1
      ha( 6) =  2

      a ( 5) =  2.0d+0
      a ( 6) = -4.0d+0

*     Don't forget to finish off  ka(n+1) = ne+1
*     This is crucial.

      ka(n+1) =  ne + 1

*     ------------------------------------------------------------------
*     Set the upper and lower bound on the row.
*     Make the linear row a free row.
*     ------------------------------------------------------------------
      bl(n+1) = -bplus
      bu(n+1) =  3.0d+0

      bl(n+2) =  -bplus
      bu(n+2) =   bplus

*     ------------------------------------------------------------------
*     Set the upper and lower bounds on the variables
*     ------------------------------------------------------------------
      do 200, j = 1, n
         bl(j) =  0.0d+0
         bu(j) =  bplus
  200 continue

*     ------------------------------------------------------------------
*     Initialize x, hs and pi.
*     Set the initial value and status of each variable.
*     For want of something better to do, make the variables x(1:n)
*     temporarily fixed at their current values. 
*     The crash can set the rest.
*     ------------------------------------------------------------------
      do 400, j = 1, n
         x (j) = 0.0d+0
         hs(j) = 0
  400 continue

      do 500, i = 1, m
         pi(i)  = 0.0d+0
  500 continue

*     end of t3data.
      end

