*     ------------------------------------------------------------------
*     File t6wood.f  (Unix version)
*     Illustrates using SNOPT on a nonlinearly constrained problem
*     with MPS data read from a file.
*
*     16 May 1998: First   version.
*     16 May 1998: Current version.
*     ------------------------------------------------------------------
      program            t6main
      implicit           double precision (a-h,o-z)

      parameter        ( maxm   = 1000,
     $                   maxn   = 1000,
     $                   maxne  = 3000 )

      character*8        PrbNms(5) , Names(maxm+maxn)
      integer            ha(maxne) , hs(maxn+maxm)
      integer            ka(maxn+1)
      double precision   a(maxne)  , bl(maxn+maxm), bu(maxn+maxm),
     $                   xs(maxn+maxm), pi(maxm)  , rc(maxn+maxm)

*     SNOPT workspace

      parameter          (  lenrw = 50000)
      double precision   rw(lenrw)
      parameter          (  leniw = 50000) 
      integer            iw(leniw)
      parameter          (  lencw =   500) 
      character*8        cw(lencw)

      logical            byname
      character*20       lfile
      external           t6con, dummy

*     ------------------------------------------------------------------
*     Specify some of the SNOPT files.
*     iSpecs  is the Specs file   (0 if none).
*     iPrint  is the Print file   (0 if none).
*     iSumm   is the Summary file (0 if none).
*
*     nout    is an output file used here by t6wood.

      iSpecs =  4
      iPrint =  15
      iSumm  =  6
      nout   =  6

      byname = .true.
 
      if ( byname ) then

*        Unix and DOS systems.  Open the Specs and print files.

         lfile = 't6wood.spc'
         open( iSpecs, file=lfile, status='OLD',     err=800 )

         lfile = 't6wood.out'
         open( iPrint, file=lfile, status='UNKNOWN', err=800 )
      end if

*     ------------------------------------------------------------------
*     First,  snInit MUST be called to initialize optional parameters 
*     to their default values.
*     ------------------------------------------------------------------
      call snInit( iPrint, iSumm,
     $             cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Read a Specs file (Optional).
*     ------------------------------------------------------------------
      call snSpec( iSpecs, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )

      if (inform .ge. 2) then
         write(nout, *) 'iSpecs > 0 but no Specs file found'
         go to 990
      end if

*     ------------------------------------------------------------------
*     Set up the data structure for the constraints.
*     MPSinp needs to know the number of nonlinear variables, etc.
*     The following calls fetch values set in the SPECS file.
*     Optionally, these values can be set in-line.
*     ------------------------------------------------------------------
      call sngeti( 'Nonlinear constraints        ', nnCon,
     $             inform, cw, lencw, iw, leniw, rw, lenrw )
      call sngeti( 'Nonlinear Jacobian  variables', nnJac,
     $             inform, cw, lencw, iw, leniw, rw, lenrw )
      call sngeti( 'Nonlinear Objective variables', nnObj,
     $             inform, cw, lencw, iw, leniw, rw, lenrw )
      call sngeti( 'MPS file                     ',  iMPS,
     $             inform, cw, lencw, iw, leniw, rw, lenrw )

*     The problem name is not needed---it is set by MPSinp. 
*     Specify the OBJECTIVE, RHS, RANGES and BOUNDS to be selected 
*     from the MPS file.  Blank names mean "select the first one".

*     PrbNms(1) = '        '    ! PROBLEM   name
      PrbNms(2) = '        '    ! OBJECTIVE name
      PrbNms(3) = '        '    ! RHS       name
      PrbNms(4) = '        '    ! RANGES    name    
      PrbNms(5) = '        '    ! BOUNDS    name

      if ( byname ) then

*        Unix and DOS systems.  Open the MPS file.

         lfile = 't6wood.mps'
         open( iMPS, file=lfile, status='OLD', err=800 )
      end if

      call MPSinp( iMPS,
     $             maxm, maxn, maxne,
     $             nnCon, nnJac, nnObj, 
     $             m, n, ne,
     $             iObj, ObjAdd, PrbNms,
     $             a, ha, ka, bl, bu, Names,
     $             hs, xs, pi,
     $             inform, nS,
     $             cw, lencw, iw, leniw, rw, lenrw )

      close( iMPS )
      if (inform .gt. 0) go to 990
      nName = m + n

*     ------------------------------------------------------------------
*     Specify any options not set in the Specs file.
*     i1 and i2 may refer to the Print and Summary file respectively.
*     Setting them to 0 suppresses printing.
*     ------------------------------------------------------------------
      itnlim = 10000
      i1     =   0
      i2     =   0
      call snseti( 'Iterations        ', itnlim, i1, i2, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Go for it, using a Cold start.
*     hs     need not be set if a basis file is to be input.
*            Otherwise, each hs(1:n) should be 0, 1, 2, 3, 4, or 5.
*            The values are used by the Crash procedure m2crsh
*            to choose an initial basis B.
*            If hs(j) = 0 or 1, column j is eligible for B.
*            If hs(j) = 2, column j is initially superbasic (not in B).
*            If hs(j) = 3, column j is eligible for B and is given
*                          preference over columns with hs(j) = 0 or 1.
*            If hs(j) = 4 or 5, column j is initially nonbasic.
*     ------------------------------------------------------------------
      call snopt ( 'Cold', m, n, ne, nName,
     $             nnCon, nnObj, nnJac,
     $             iObj, ObjAdd, PrbNms(1),
     $             t6con, dummy,
     $             a, ha, ka, bl, bu, Names,
     $             hs, xs, pi, rc, 
     $             inform, mincw, miniw, minrw,
     $             nS, nInf, sInf, Obj,
     $             cw, lencw, iw, leniw, rw, lenrw,
     $             cw, lencw, iw, leniw, rw, lenrw )

      if (inform .eq. 42 .or. inform .eq. 43 .or. inform .eq. 44) then
         write(nout, *) ' '
         write(nout, *) 'Estimate of required lencw:', mincw
         write(nout, *) 'Estimate of required leniw:', miniw
         write(nout, *) 'Estimate of required lenrw:', minrw
         go to 910
      end if

      write(nout, *) ' '
      write(nout, *) 'snopt finished.'
      write(nout, *) 'inform =', inform
      write(nout, *) 'nInf   =', nInf
      write(nout, *) 'sInf   =', sInf
      write(nout, *) 'Obj    =', Obj
      if (inform .ge. 20) go to 910
      stop

*     ------------------------------------------------------------------
*     Error exit.
*     ------------------------------------------------------------------
  800 write(nout, 4000) 'Error while opening file', lfile
      stop

  910 write(nout, *) ' '
      write(nout, *) 'STOPPING because of error condition'

  990 stop

 4000 format(/  a, 2x, a  )

*     end of t6main.
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine t6con ( mode, m, n, neJac, x, f, g, nState,
     $                   cw, lencw, iw, leniw, rw, lenrw )

      implicit           double precision (a-h,o-z)
      double precision   x(n), f(m), g(m,n)

      character*8        cw(lencw)
      integer            iw(leniw)
      double precision   rw(lenrw)

*     ------------------------------------------------------------------
*     t6con  is funcon for test problem t6wood,
*     a chemical engineering design problem.
*     Originally called  woplant  (wood plant?).
*     m = 5,  n = 10.
*
*     For test purposes, we test  Derivative level
*     to decide whether or not to compute gradients.
*
*     Dec 1981: Original SNOPT version obtained via Bruce Murtagh.
*     Oct 1991: Converted to f77 for test problem t6wood.
*     ------------------------------------------------------------------
      parameter (one   = 1.0d+0,  two  = 2.0d+0,
     $           three = 3.0d+0,  four = 4.0d+0,
     $           half  = 0.5d+0,  tenk = 10000.0d+0,  vrho = 3000.0d+0)
*     ------------------------------------------------------------------
      iPrint    = iw( 12) ! Print file
      iSumm     = iw( 13) ! Summary file
      lvlDer    = iw( 70) ! = 0, 1, 2 or 3, the derivative level

      if (nState .eq. 1) then
         if (iPrint .gt. 0) write(iPrint, 5) lvlDer
         if (iSumm  .gt. 0) write(iSumm , 5) lvlDer
    5    format(/ ' This is problem t6wood.  Derivative level =', i3 /)
      end if

*     Transform to original variables.

      fg     = tenk*(one + x(1))
      fp     = tenk*(one + x(2))
      fd     = tenk*(one + x(3))
      fra    = tenk*(one + x(4))
      frp    = tenk*(one + x(5))
      fre    = tenk*(one + x(6))
      frb    = tenk*(one + x(7))
      frc    = tenk*(one + x(8))
      fr     = tenk*(one + x(9))
      temp   = 630.0d+0 + 50.0d+0*x(10)

*     Rate constants.

      ak1    = 5.9755d+09 * dexp(-1.2d+4/temp)
      ak2    = 2.5962d+12 * dexp(-1.5d+4/temp)
      ak3    = 9.6283d+15 * dexp(-2.0d+4/temp)

*     Rate terms.

      fr2    = fr**2
      r1     = ak1*fra*frb*vrho/fr2
      r2     = ak2*frb*frc*vrho/fr2
      r3     = ak3*frc*frp*vrho/fr2

*     Nonlinear functions.

      recip  = one/(fr - fg - fp)
      f(1)   = two*r2       - fd*recip*fre
      f(2)   = r2 - half*r3 - fd*recip*(frp - fp) - fp
      f(3)   = - r1         - fd*recip*fra
      f(4)   = - r1 - r2    - fd*recip*frb
      f(5)   = 1.5d+0*r3    - fg

*     Scale them.

      do 10 i = 1, m
         f(i) = f(i) / tenk
   10 continue

*     Compute the Jacobian (if SNOPT wants it).

      if (mode .eq. 0  .or.  lvlDer .lt. 2) return

      b1t    = 1.2d+4/temp**2
      b2t    = 1.5d+4/temp**2
      b3t    = 2.0d+4/temp**2
      rr     = recip**2

      g(1,1) = - fd*fre*rr
      g(1,2) =   g(1,1)
      g(1,3) = - fre*recip
      g(1,6) = - fd *recip
      g(1,7) =   two*r2/frb
      g(1,8) =   two*r2/frc
      g(1,9) = - four*r2/fr - g(1,1)
      g(1,10)=   two*r2*b2t

      g(2,1) = - fd*(frp - fp)*rr
      g(2,2) =   fd*(fr - frp - fg)*rr - one
      g(2,3) = - (frp - fp)*recip
      g(2,5) = - half*r3/frp - fd*recip
      g(2,7) =   r2/frb
      g(2,8) =   (r2 - half*r3)/frc
      g(2,9) = - two*(r2 - half*r3)/fr - g(2,1)
      g(2,10)=   r2*b2t - half*r3*b3t

      g(3,1) = - fd*fra*rr
      g(3,2) =   g(3,1)
      g(3,3) = - fra*recip
      g(3,4) = - r1/fra - fd*recip
      g(3,7) = - r1/frb
      g(3,9) =   two*r1/fr - g(3,1)
      g(3,10)= - r1*b1t

      g(4,1) = - fd*frb*rr
      g(4,2) =   g(4,1)
      g(4,3) = - frb*recip
      g(4,4) = - r1/fra
      g(4,7) = - (r1 + r2)/frb - fd*recip
      g(4,8) = - r2/frc
      g(4,9) =   two*(r1+r2)/fr - g(4,1)
      g(4,10)= - r1*b1t - r2*b2t

      g(5,1) = - 1.0d+0
      g(5,5) =   1.5d+0*r3/frp
      g(5,8) =   1.5d+0*r3/frc
      g(5,9) = - three *r3/fr
      g(5,10)=   1.5d+0*r3*b3t

*     Rescale the temperature derivatives.

      do 50 i = 1, m
         g(i,10) = g(i,10) * 5.0d-3
   50 continue

*     end of t6con  (funcon for Chemical Design Problem woplant)
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine dummy ( mode, n, x, f, g, nState,
     $                   cw, lencw, iw, leniw, rw, lenrw )

      implicit           double precision (a-h,o-z)
      double precision   x(n), g(n)

      character*8        cw(lencw)
      integer            iw(leniw)
      double precision   rw(lenrw)

*     ==================================================================
*     Problem t6wood.
*     No nonlinear objective.
*     ==================================================================

*     Relax

*     end of dummy objective for t6wood
      end

