*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
*     file  sn65rmod.f
*
*     s6BFGS   s6chol   s6Radd   s6Rcnd   s6Rcyc   s6Rmod   s6Rsol
*     s6Rswp   s6Rupd
*
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine s6BFGS( lenR, n, nFull, ydx, dxHdx, R, Hdx, y, Rdx )

      implicit
     &     none
      integer
     &     lenR, n, nFull
      double precision
     &     ydx, dxHdx, R(lenR), Hdx(n), y(n), Rdx(n)

*     ==================================================================
*     s6BFGS  applies the quasi-Newton update to the Cholesky factor R
*     of the approximate Hessian.
*
*     On entry:
*     R      contains the first n rows and columns of an
*            upper-triangular matrix R such that R'R = H.
*
*     Hdx    contains H times the vector x2 - x1.
*            It is overwritten. 
*
*     y      contains the first  n  components of the difference g2 - g.
*
*     Rdx    is the vector such that  R'*(Rdx) = Hdx.
*            It is overwritten.
*
*     w      is a work vector.
*
*     On exit:
*
*     30 Dec 1991: First version based on  Npsol  routine npbfgs.
*     05 May 1999: Current version.
*     ==================================================================
      integer
     &     k, lastnz
      double precision
     &     c, rtydx, tolz
*     ------------------------------------------------------------------
      double precision   zero,          one
      parameter         (zero = 0.0d+0, one = 1.0d+0)
*     ------------------------------------------------------------------
      tolz   = zero
      rtydx  = sqrt( ydx )

*     Normalize  Rdx.
*     Scale Hdx.

      c      = one/sqrt(dxHdx)
      call dscal ( n, (-c), Hdx, 1 )
      call dscal ( n,   c , Rdx, 1 )
 
*     ------------------------------------------------------------------
*     Apply the update to R. 
*     Set w for the update  R + v w'.
*     w is held in Hdx,  v  is held in Rdx.
*     ------------------------------------------------------------------
      call daxpy ( n, (one/rtydx), y, 1, Hdx, 1 )

      if (n .lt. nFull) then
*        --------------------------------------------
*        The leading nxn section of R is updated.
*        --------------------------------------------
         lastnz = n + 1
      else
*        --------------------------------------------
*        The full R is updated.
*        Find the last nonzero in v.
*        --------------------------------------------
         do k = n, 1, -1
            lastnz = k
            if (abs( Rdx(lastnz) ) .gt. tolz) go to 500
         end do
      end if
*     -------------------------------------------------------
*     Restore  R + v w'  to triangular form  (overwriting v).
*     -------------------------------------------------------
  500 call s6Rmod( n, lenR, R, Rdx, Hdx, lastnz, one, tolz )

      end ! of s6BFGS

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine s6chol( pivot,
     &     inform, n, Hdmin, dmax, iRank, perm, lenH, H )

      implicit
     &     none
      integer
     &     pivot, inform, n, iRank, lenH, perm(n)
      double precision
     &     Hdmin, dmax, H(lenH)

*     ==================================================================
*     s6chol  forms the upper-triangular Cholesky factor R  such that 
*                            H = R'R.
*
*     On entry,
*        pivot  specifies the pivoting strategy. 
*               pivot = 'N'o  specifies no pivoting, otherwise
*                       complete (i.e., diagonal) pivoting is used.
*
*        Hdmin  is the magnitude of the square of the smallest 
*               allowable diagonal of R.
*
*     On exit, 
*        perm   contains details of the permutation matrix P, such that
*               perm( k ) = k  if no column interchange occured at the
*               the kth step and  perm( k ) = j, ( k .lt. j .le. n ), 
*               if columns  k and j  were interchanged at the kth step.
*
*     Only the diagonal and super-diagonal elements of H are used,
*     and they are overwritten by the Cholesky factor, R.
*
*     28 May 1994: First version of s6chol based on routine chlfac.
*     15 Oct 2000: Current version.
*     ==================================================================
      integer
     &     i, is, ix, j, js, jx, k, kmax,
     &     l, lmax, ls, lcolj, lcolk, lcj, lck, lrj, lrk
      double precision
     &     d, dmin, Hjx, s
*     ------------------------------------------------------------------
      integer            NoPiv,      Piv
      parameter         (NoPiv  = 0, Piv    = 1)
      double precision   zero
      parameter         (zero = 0.0d+0)
*     ------------------------------------------------------------------
      inform = 0
      iRank  = 0
      if (n .eq. 0  .or. (pivot .ne. Piv  .and.  pivot .ne. NoPiv)) then
         return
      end if

      dmin   = Hdmin

*     ------------------------------------------------------------------
*     Form the Cholesky factorization R'R = Z'HZ.
*     Process the first n rows of H.
*     Use symmetric interchanges if requested.
*     ------------------------------------------------------------------
      l = 1
      do j = 1, n
         dmax = abs( H(l) )
         kmax = j
         lmax = l 

         if (pivot .eq. Piv) then

*           Find the diagonal of the Schur complement with maximum
*           absolute value.

            ls = l + j + 1 
            do k = j+1, n
               if (dmax .lt. abs( H(ls) )) then
                  dmax = abs( H(ls) )
                  kmax = k
                  lmax = ls
               end if
               ls = ls + k + 1
            end do
         end if

         dmax   = H(lmax)

*        Check that the diagonal is big enough.

         if (dmax .le. dmin) go to 800

*        If necessary, perform a symmetric interchange.

         perm(j) = kmax

         if (kmax .ne. j) then

            lcolj = l - j + 1
            lcolk = kmax*(kmax - 1)/2 + 1

*           call dswap ( kmax-j, H(j+1,kmax), 1, H(j,j+1), ldH )

            lck  = lcolk + j
            lrj  = l     + j
            do i = j+1, kmax
               s      = H(lrj) 
               H(lrj) = H(lck)  
               H(lck) = s 
               lrj    = lrj + i
               lck    = lck + 1
            end do

*           call dswap ( j, H(1,j), 1, H(1,kmax), 1 )

            lcj  = lcolj
            lck  = lcolk
            do i = 1, j
               s      = H(lcj) 
               H(lcj) = H(lck)  
               H(lck) = s 
               lcj    = lcj + 1
               lck    = lck + 1
            end do

*           call dswap ( n-kmax+1, H(kmax,kmax), ldH, H(j,kmax), ldH )

            lrk  = lcolk + kmax - 1
            lrj  = lcolk + j    - 1
            do i = kmax, n
               s      = H(lrj) 
               H(lrj) = H(lrk)  
               H(lrk) = s 
               lrj    = lrj + i
               lrk    = lrk + i
            end do
         end if ! kmax ne j

*        Set the diagonal of  R.

         d     = sqrt( dmax )
         H(l)  = d
         iRank = iRank + 1

         if (j .lt. n) then

*           Set the super-diagonal elements of the jth row of R.
*           Do a rank-one update to the Schur complement.

            jx = l + j
            do k = j+1, n
               H(jx) = H(jx)/d
               jx    = jx + k
            end do

            jx = l + j
            ls = l + j + 1 

            do js = j+1, n

*              Form the upper-triangular part of 
*                 H(:,js) = H(:,js) - H(j,js)*H(:,j).

               if (H(jx) .ne. zero) then
                  Hjx = H(jx)
                  ix  = l + j
                  do is = j+1, js
                     H(ls) = H(ls) - Hjx*H(ix)
                     ix    = ix + is
                     ls    = ls + 1
                  end do
               end if
               jx = jx + js
               ls = jx + 1
            end do
         end if
         l = l + j + 1
      end do

*     ==================================================================
*     Test if  H  is not positive definite.
*     ==================================================================
  800 if (iRank .lt. n) inform = 1

      end ! of s6chol

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine s6Radd( n, lenR, R, v, tolz )

      implicit
     &     none
      integer
     &     n, lenR
      double precision
     &     tolz, R(lenR), v(n)

*     ==================================================================
*     s6Radd  modifies the first n rows and columns of the n x n
*     upper-triangular matrix  R  so that  Q* ( R  )  is upper
*                                             ( v' )
*     triangular, where  Q  is orthogonal.
*     v  is an n-vector whose elements are stored in the array v.
*     The new  R  overwrites the old.
*
*     Q  is the product of plane rotations (not stored).
*     The array   v  is overwritten.
*
*     firstv points to the first nonzero of  v.  The value firstv = 1
*            would always be ok, but sometimes it is known to be bigger
*            than  1,  so  Q  is a partial sweep of rotations.
*
*     tolz   is a tolerance for negligible elements in  v.
*
*     27 Nov 1998: First version of s6Radd.
*     28 May 1999: Current version.
*     ==================================================================
      integer
     &     i, j, l, ldiag
      double precision
     &     cs, root, s, sn, t
*     ------------------------------------------------------------------
*     Eliminate the elements of  v  using a forward sweep of rotations.

      if (n .gt. 0) then
         ldiag  = 0

         do i = 1, n
            ldiag = ldiag + i
            t     = v(i)
            if (abs(t) .gt. tolz) then
               s        = R(ldiag)
               root     = sqrt(s**2 + t**2)
               cs       = s/root
               sn       = t/root
               R(ldiag) = root
               l        = ldiag + i

               do j = i+1, n
                  s    = R(l)
                  t    = v(j)
                  R(l) = cs*s + sn*t
                  v(j) = sn*s - cs*t
                  l    = l + j
               end do
            end if
         end do
      end if

      end ! of s6Radd

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine s6Rcnd( nR, lenR, R, dRmax, dRmin, Rmax, condH )

      implicit
     &     none
      integer
     &     nR, lenR
      double precision
     &     dRmax, dRmin, Rmax, condH, R(lenR)

*     ==================================================================
*     s6Rcnd  finds the largest and smallest diagonals of the
*     upper triangular matrix  R, and returns the square of their ratio.
*     This is a lower bound on the condition number of  R' R.
*
*     28 May 1999: Current version.
*     ==================================================================
      logical
     &     overfl
      integer
     &     i, j, l
      double precision
     &     d, ddiv
*     ------------------------------------------------------------------
      double precision   one
      parameter         (one = 1.0d+0)
*     ------------------------------------------------------------------
      overfl    = .false.

      if (nR .eq. 0) then     
         condH = one
         dRmin = one
         dRmax = one
         Rmax  = one
      else
         dRmax = abs( R(1) )
         dRmin = dRmax
         Rmax  = dRmax

         if (nR .eq. 1) then     
            condH = ddiv( one, dRmin**2, overfl )
            if (condH .lt. one) condH = one/condH
         else 
            l     = 2
            do j = 2, nR
               do i = 1, j
                  d    = abs( R(l) )
                  Rmax = max( Rmax, d )
                  l    = l + 1
               end do

               dRmin = min( dRmin, d )
               dRmax = max( dRmax, d )
            end do
            
            condH  = (dRmax/dRmin)**2
         end if
      end if

      end ! of s6Rcnd

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine s6Rcyc( jq, jr, eps, lenR, nR, R, w )

      implicit
     &     none
      integer
     &     jq, jr, lenR, nR
      double precision
     &     eps, R(lenR), w(nR)

*     ==================================================================
*     s6Rcyc  moves column jq of the upper-triangular matrix R
*     to position jr, and cyclically shifts the intervening columns
*     to the right or left (depending on whether jq > jr or jq < jr).
*     R is retriangularized using a partial sweep of plane rotations.
*
*     13 Aug 1992: First version of s6Rcyc.
*     28 May 1999: Current version.
*     ==================================================================
      integer
     &     i, j, jq0, jq1, jr0, jr1, k, l, lqdiag, lrdiag, lr, lt, lv,
     &     lw, lz
      double precision
     &     d, sn, cs, root, s, t, tolz, v
*     ------------------------------------------------------------------
      double precision   zero
      parameter        ( zero = 0.0d+0 )
*     ------------------------------------------------------------------
      tolz   = eps
      tolz   = zero

      jq0    = jq - 1
      jr0    = jr - 1
      jq1    = jq + 1
      jr1    = jr + 1
      lqdiag = jq * jq1 / 2
      lrdiag = jr * jr1 / 2

      if (jq .gt. jr) then
*        ===============================================================
*        jq > jr.  Move column jq to the left.
*
*        We first permute the middle rows and columns of R
*        from  ( R1  S   w1  T1 )  to  ( R1  w1  S   T1 ),
*              (     R2  w2  T2 )      (     d   0   v' )
*              (         d   v' )      (     w2  R2  T2 )
*              (             R3 )      (             R3 )
*        where columns jq and jr of the initial R correspond to
*        w1 and the first column of S respectively.
*
*        We then reduce w2 to zero with a backward sweep of rotations
*        into the new row jr, altering (d  0  v') and (R2  T2).        
*        ===============================================================
*        Extract d, w1 and w2.
*        w' contains  ( w1'  d  w2' )

         d    = R(lqdiag)
         lr   = lqdiag - 1
         lw   = jq

         do i = jq0, 1, -1
            w(lw) = R(lr)
            lr    = lr - 1
            lw    = lw - 1
            if (i .eq. jr) lw = lw - 1
         end do

*        ---------------------------------------------------------------
*        Move R2 and S, running backwards up the columns.
*        ---------------------------------------------------------------
         k      = lqdiag - jq
         l      = lqdiag

         do j = jq0, jr, -1
            do i = j, 1, -1
               R(l) = R(k)
               k    = k - 1
               l    = l - 1
               if (i .eq. jr) l = l - 1
            end do
         end do

*        Put w1 in between R1 and S.

         lr     = lrdiag - jr0
         call dcopy ( jr0, w, 1, R(lr), 1 )

*        ---------------------------------------------------------------
*        Change  ( T2 )  to  ( v' ).
*                ( v' )      ( T2 )
*        ---------------------------------------------------------------
         lv     = lqdiag + jq

         do j = jq1, nR
            v  = R(lv)
            lt = lv

            do i = jq, jr1, -1
               R(lt) = R(lt - 1)
               lt    =   lt - 1
            end do

            R(lt) = v
            lv    = lv + j
         end do

*        ---------------------------------------------------------------
*        Triangularize  ( d   0   v' )  using a partial backward sweep
*                       ( w2  R2  T2 )
*        of rotations to eliminate w2.
*        lv and lr mark the start of the rows involved.
*        ---------------------------------------------------------------
         lv = lqdiag - jq + jr
         lr = lqdiag

         do i = jq, jr1, -1
            t    = w(i)
            root = sqrt( d**2 + t**2 )
            cs   = d/root
            sn   = t/root

*           Apply rotation to rows jr and i.

            R(lv) = zero
            k     = lv
            l     = lr

            if (abs(sn) .gt. tolz) then
               d = root
               do j = i, nR
                  s    = R(k)
                  t    = R(l)
                  R(k) = cs*s + sn*t
                  R(l) = sn*s - cs*t
                  k    = k + j
                  l    = l + j
               end do
            end if

            lv = lv - i + 1
            lr = lr - i
         end do

*        ---------------------------------------------------------------
*        Store the final diagonal.
*        ---------------------------------------------------------------
         R(lrdiag) = d

      else if (jq .lt. jr) then
*        ===============================================================
*        jq < jr.  Move column jq to the right.
*
*        We first permute the middle rows and columns of R
*        from  ( R1  w1  S   T1 )  to  ( R1  S   w1  T1 ),
*              (     d   w2' w3')      (     R2  0   T2 )
*              (         R2  T2 )      (     w2' d   w3')
*              (             R3 )      (             R3 )
*        where columns jq and jr of the initial R correspond to
*        w1 and the last column of S respectively.
*
*        We then reduce w2' to zero with a forward sweep of rotations
*        into the new row jq, altering (R2  0  T2) and (d  w3').
*        ===============================================================
*        Extract d, w1, w2 and w3.
*        w' contains  ( w1'  w2'  d  w3' )

         d  = R(lqdiag)
         lr = lqdiag - jq0
         call dcopy ( jq0, R(lr), 1, w, 1 )

         lr = lqdiag + jq
         lw = jq

         do j = jq1, nR
            w(lw) = R(lr)
            lr    = lr + j
            lw    = lw + 1
            if (j .eq. jr) lw = lw + 1
         end do

         w(jr) = d

*        ---------------------------------------------------------------
*        Move S and R2, running forwards down the columns.
*        ---------------------------------------------------------------
         l = lqdiag - jq0
         k = lqdiag + 1

         do j = jq1, jr
            do i = 1, j-1
               if (i .eq. jq) k = k + 1
               R(l) = R(k)
               l    = l + 1
               k    = k + 1
            end do
         end do

*        Put w1 in between S and T1.

         lr = lrdiag - jr0
         call dcopy ( jq0, w, 1, R(lr), 1 )

*        ---------------------------------------------------------------
*        Move T2 up one row.
*        ---------------------------------------------------------------
         lr = lrdiag + jq

         do j = jr1, nR
            lt = lr
            do i = jq, jr0
               R(lt) = R(lt + 1)
               lt    =   lt + 1
            end do

            lr = lr + j
         end do

*        ---------------------------------------------------------------
*        Triangularize  ( R2  0  T2 )  using a partial backward sweep
*                       ( w2' d  w3')
*        of rotations to eliminate w2'.
*        lr marks the start of the rows of R2.
*        lz marks the zero to be inserted between R2 and T2.
*        ---------------------------------------------------------------
         lr = lqdiag
         lz = lrdiag - jr + jq

         do i = jq, jr0
            R(lz) = zero
            d     = R(lr)
            t     = w(i)
            root  = sqrt( d**2 + t**2 )
            cs    = d/root
            sn    = t/root

*           Apply rotation to rows i and jr  (where row jr is in w').

            k = lr + i

            if (abs(sn) .gt. tolz) then
               R(lr) = root
               do j = i+1, nR
                  s    = R(k)
                  t    = w(j)
                  R(k) = cs*s + sn*t
                  w(j) = sn*s - cs*t
                  k    = k + j
               end do
            end if

            lr = lr + i + 1
            lz = lz + 1
         end do

*        ---------------------------------------------------------------
*        Store w(jr) ... w(nR) as the final jr-th row of R.
*        ---------------------------------------------------------------
         lr = lrdiag

         do j = jr, nR
            R(lr) = w(j)
            lr    = lr + j
         end do
      end if

      end ! of s6Rcyc

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine s6Rmod( n, lenR, R, v, w, lastv, vnorm, tolz )

      implicit
     &     none
      integer
     &     n, lenR, lastv
      double precision
     &     vnorm, tolz, R(lenR), v(n), w(n)

*     ==================================================================
*     s6Rmod  modifies the first n rows and columns of the (n+1)x(n+1)
*     upper-triangular matrix  R  so that  Q*(R + v w')  is upper
*     triangular, where  Q  is orthogonal.
*     v  and  w  are (n+1)-vectors whose first n elements are stored in
*     arrays v and w.  The last component of v is held in vlast.
*     The new  R  overwrites the old.
*
*     Q  is the product of two sweeps of plane rotations (not stored).
*     These affect the  (lastv)th  row of  R,  which is temporarily held
*     in the array  v.  Thus,  v  is overwritten.  w  is not altered.
*
*     lastv  points to the last nonzero of  v.  The value lastv = n
*            would always be ok, but sometimes it is known to be less
*            than  n,  so  Q  reduces to two partial sweeps of
*            rotations.
*
*     vnorm  is the norm of the (n+1)-vector v.
*
*     tolz   is a tolerance for negligible elements in  v.
*
*     06 Sep 1991: First version based on Minos routine m6rmod.
*     11 Sep 1994: Modified to update a leading section of R.
*     28 May 1998: Current version.
*     ==================================================================
      integer
     &     i, j, l, lastR, ldiag
      double precision
     &     dnrm2, root, cs, s, sn, t, v2, vlast
*     ------------------------------------------------------------------
      double precision   zero
      parameter        ( zero = 0.0d+0 )
*     ------------------------------------------------------------------

      lastR  = lastv*(lastv + 1)/2

      if (lastv .le. n) then
         vlast = v(lastv)
      else
         vlast = sqrt( max(zero, vnorm**2 - dnrm2( n, v, 1 )**2) )
      end if

*     Copy the (lastv)th row of  R  into the end of  v.

      l  = lastR
      do j = lastv, n
         v(j) = R(l)
         l    = l + j
      end do

*     ------------------------------------------------------------------
*     Reduce v to a multiple of e(lastv) using a partial backward sweep
*     of rotations.  This fills in the (lastv)th row of R (held in v).
*     ------------------------------------------------------------------
      if (lastv .gt. 1) then
         v2     = vlast**2
         ldiag  = lastR

         do i = lastv-1, 1, -1
            ldiag = ldiag - i - 1
            s     = v(i)
            v(i)  = zero
            if (abs(s) .gt. tolz) then
               v2    = s**2 + v2
               root  = sqrt(v2)
               cs    = vlast/root
               sn    = s    /root
               vlast = root
               l     = ldiag

               do j = i, n
                  s    = v(j)
                  t    = R(l)
                  v(j) = cs*s + sn*t
                  R(l) = sn*s - cs*t
                  l    = l + j
               end do
            end if
         end do
      end if

*     Set  v  =  v  +  vlast*w.

      call daxpy ( n, vlast, w, 1, v, 1 )

*     ------------------------------------------------------------------
*     Eliminate the front of the (lastv)th row of R (held in v) using a
*     partial forward sweep of rotations.
*     ------------------------------------------------------------------
      if (lastv .gt. 1) then
         ldiag  = 0

         do i = 1, lastv-1
            ldiag = ldiag + i
            t     = v(i)
            if (abs(t) .gt. tolz) then
               s        = R(ldiag)
               root     = sqrt(s**2 + t**2)
               cs       = s/root
               sn       = t/root
               R(ldiag) = root
               l        = ldiag + i

               do j = i+1, n
                  s    = R(l)
                  t    = v(j)
                  R(l) = cs*s + sn*t
                  v(j) = sn*s - cs*t
                  l    = l + j
               end do
            end if
         end do
      end if

*     -----------------------------------
*     Insert the new (lastv)th row of  R.
*     -----------------------------------
      l  = lastR
      do j = lastv, n
         R(l) = v(j)
         l    = l + j
      end do

      end ! of s6Rmod

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine s6Rsol( Task, lenR, nR, R, y )

      implicit
     &     none
      integer
     &     Task, lenR, nR
      double precision
     &     R(lenR), y(nR)

*     ==================================================================
*     s6Rsol  solves  R x = y  or  R' x = y,  where  R  is an
*     upper-triangular matrix stored by columns in  R(lenR).
*     The solution  x  overwrites  y.
*     ==================================================================
      integer
     &     i, ii, l
      double precision
     &     ddot, t
*     ------------------------------------------------------------------
      integer            WithR,      WithRt
      parameter         (WithR  = 0, WithRt = 1)
*     ------------------------------------------------------------------
      if (Task .eq. WithR) then

*        Solve R y = y.

         l  = nR*(nR + 1)/2
         do ii = 1, nR
            i    = nR + 1 - ii
            t    = y(i)/R(l)
            y(i) = t
            if (i .ne. 1) then
               l = l - i
               call daxpy ( i-1, (-t), R(l+1), 1, y, 1 )
            end if
         end do

      else if (Task .eq. WithRt) then

*        Solve Rtranspose y = y.

         y(1) = y(1)/R(1)
         if (nR .gt. 1) then
            l = 1
            do i = 2, nR
               t    = ddot  ( i-1, R(l+1), 1, y, 1 )
               l    = l + i
               y(i) = (y(i) - t)/R(l)
            end do
         end if
      end if

      end ! of s6Rsol

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine s6Rswp( nR, lenR, R, v, w, lastv )

      implicit
     &     none
      integer
     &     nR, lenR, lastv
      double precision
     &     R(lenR), v(nR), w(nR)

*     ==================================================================
*     s6Rswp  modifies the upper-triangular matrix R to account for a 
*     basis exchange in which the  (lastv)th superbasic variable becomes
*     basic.  R  is changed to R  +  vw',  which is triangularized by
*     s6Rmod.  v  is the  (lastv)th column of R,  and w is input.
*
*     01 Dec 1991: First version based on Minos routine m6bswp.
*     24 Apr 1994: Nx no longer in Q.
*     24 Jan 1996: v left unscaled.
*     28 May 1998: Current version of s6Rswp.
*     ==================================================================
      integer
     &     l
      double precision
     &     dasum, tolz, vnorm
*     ------------------------------------------------------------------
      double precision   zero
      parameter         (zero   = 0.0d+0)
*     ------------------------------------------------------------------
      tolz   = zero

*     Set  v =  lastv-th column of  R  and find its norm.

      l      = (lastv - 1)*lastv/2
      call dcopy ( lastv, R(l+1), 1, v, 1 )
      vnorm  = dasum ( lastv, v, 1 )
      call s6Rmod( nR, lenR, R, v, w, lastv, vnorm, (vnorm*tolz) )

      end ! of s6Rswp

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine s6Rupd( Update, lenR, m, n, nBS, nnL, nS,
     &     H0scal, ydx, dxHdx,
     &     ne, nlocA, locA, indA, Acol,
     &     kBS, gdif, Hdx, R, w, y, y2,
     &     iw, leniw, rw, lenrw )

      implicit
     &     none
      integer
     &     Update, lenR, m, n, nBS, ne, nlocA, nnL, nS, leniw, lenrw,
     &     indA(ne), locA(nlocA), kBS(nBS), iw(leniw)
      double precision
     &     H0scal, ydx, dxHdx, Acol(ne), R(lenR), gdif(nnL),
     &     Hdx(nnL), y(nBS), y2(nBS), w(nBS), rw(lenrw)
      
*     ==================================================================
*     s6Rupd updates the Cholesky factor of the reduced Hessian.
*     
*     04 Aug 1995: First version based on NPSOL routine npupdt.
*     20 Jul 2000: Current version of s6Rupd.
*     ==================================================================
      integer
     &     inform, lenR0
      double precision
     &     eps0, rH0scl 
*     ------------------------------------------------------------------
      integer            Gather
      parameter         (Gather = 0)
      integer            WithBt   
      parameter         (WithBt = 2)
      integer            Transp,     WithRt     
      parameter         (Transp = 1, WithRt = 1)
      double precision   one
      parameter         (one = 1.0d+0)
*     ------------------------------------------------------------------
      eps0      = rw(  2)
     
*     ------------------------
*     Find Z'*gdif and  Z'Hdx.
*     ------------------------
      call s2copy( Gather, nnL, nBS, kBS, one, gdif, y  )
      call s2Bsol( WithBt, inform, m, y, w, iw, leniw, rw, lenrw )

      if (nS .gt. 0) then
         call s2Bprd( Transp, eps0, n, nS, kBS(m+1),
     &        ne, nlocA, locA, indA, Acol,
     &        (-one), w, m, one, y(m+1), nS )
      end if

      call s2copy( Gather, nnL, nBS, kBS, one, Hdx , y2 )
      call s2Bsol( WithBt, inform, m, y2, w, iw, leniw, rw, lenrw )

      if (nS .gt. 0) then
         call s2Bprd( Transp, eps0, n, nS, kBS(m+1),
     &        ne, nlocA, locA, indA, Acol,
     &        (-one), w, m, one, y2(m+1), nS ) 
      end if

      if (Update .gt. 1) then
*        --------------------
*        Self-scaled BFGS.
*        --------------------
         rH0scl    = sqrt( H0scal )
         lenR0     = nS*(nS + 1)/2
         call dscal( lenR0, rH0scl, R , 1 )                
      end if

*     ------------------------------------------------------------------
*     Apply the BFGS update to the first nS columns of R.
*     y = Z'gdif,  y2 = Rdx  and  w = Z'Hdx.
*     ------------------------------------------------------------------
      call dcopy ( nS, y2(m+1), 1, w(m+1), 1 )
      call s6Rsol( WithRt, lenR, nS, R, y2(m+1) )
      call s6BFGS( lenR, nS, nnL, ydx, dxHdx, 
     &     R, w(m+1), y(m+1), y2(m+1) ) 

      end ! of s6Rupd

