*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
*     File  sn90lmqn.f.   Limited-memory BFGS routines.
*
*     s9LMH0   s9LMup   s9LMHx
*
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine s9LMH0( Htype, nnH, mQNmod, nQNmod,
     &     H0pre, H0, gdif, Hxdif, ytdx, dxtHdx )

      implicit
     &     none
      integer
     &     Htype, nnH, mQNmod, nQNmod
      double precision
     &     H0pre, H0(nnH), gdif(nnH,mQNmod), Hxdif(nnH,mQNmod),
     &     ytdx(mQNmod), dxtHdx(mQNmod)

*     ==================================================================
*     s9LMH0 resets the approximate Hessian H to a diagonal matrix.
*     If H is already diagonal it is set to the identity.
*     On entry, the value of Htype is as follows:
*
*       Htype 
*       -----
*       HUnset (-1)      H not set.
*       HNorml ( 0)      H is a Hessian of the form defined by  lvlHes.
*       HDiag  ( 1)      H is a diagonal matrix.
*       HUnit  ( 2)      H is an identity matrix.
*
*     19 Jul 1995: First version of s9LMH0 written by PEG.
*     06 Sep 1998: Pre- and post-QP diagonal Hessian scaling added.
*     07 Nov 2000: Current version.
*     ==================================================================
      integer
     &     i, k
      double precision
     &     sclObj, H0min
*     ------------------------------------------------------------------
      integer            HNorml,     HDiag,      HUnit 
      parameter         (HNorml = 0, HDiag  = 1, HUnit  = 2)
      double precision   zero,             one
      parameter         (zero   =  0.0d+0, one    = 1.0d+0)
*     ------------------------------------------------------------------
      if (H0pre .ge. zero) then
         sclObj =   one
      else
         sclObj = - one
      end if

      if (Htype .eq. HNorml) then
*        ------------------------------------------------------------
*        Reset H0 to the diagonal of the current H.
*        If H0 is indefinite, use  H0 = H0pre*I
*        ------------------------------------------------------------
         H0min = one
         do  k = 1, min(mQNmod, nQNmod)
            do i = 1, nnH
               H0(i) = H0(i) - Hxdif(i,k)*Hxdif(i,k)*dxtHdx(k)
     &                       +  gdif(i,k)* gdif(i,k)*  ytdx(k)
               H0min = min(sclObj*H0(i), H0min)
            end do
         end do
         if (H0min .le. zero) Htype  = HUnit
      end if

      if (Htype .eq. HNorml) then
         Htype  = HDiag
      else
*        ------------------------------------------------------------
*        Set  H0 = H0pre*I.
*        ------------------------------------------------------------
         call dload ( nnH, H0pre, H0, 1 )
         Htype  = HUnit
      end if

      nQNmod = 0

      end ! of s9LMH0

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine s9LMup( Update, Htype, nnH, mQNmod, nQNmod,
     &     H0pre, H0scal, ydx, dxHdx, sgnObj, Hdx, y,
     &     H0, gdif, Hxdif, ytdx, dxtHdx )

      implicit
     &     none
      integer
     &     Update, Htype, nnH, mQNmod, nQNmod
      double precision
     &     H0pre, H0scal, sgnObj, ydx, dxHdx, Hdx(nnH), y(nnH),
     &     ytdx(mQNmod), dxtHdx(mQNmod), H0(nnH), gdif(nnH,mQNmod),
     &     Hxdif(nnH,mQNmod)

*     ==================================================================
*     s9LMup does almost everything associated with the limited-memory 
*     quasi-Newton update.
*     If defined, the self-scaling BFGS update parameter is saved.
*     It is needed to update the reduced Hessian when there are only
*     linear constraints.
*
*     19 Jul 1995: First version of s9LMup written by PEG.
*     06 Sep 1998: Pre- and post-QP diagonal Hessian scaling added.
*     24 Sep 1999: Current version.
*     ==================================================================
      integer
     &     i, k
      double precision
     &     ydxI, dxHdxI
*     ------------------------------------------------------------------
      integer            HNorml,     HUnit 
      parameter         (HNorml = 0, HUnit  = 2)
      double precision   one
      parameter         (one = 1.0d+0)
*     ------------------------------------------------------------------
      ydxI   = one / (sgnObj*ydx)
      dxHdxI = one / (sgnObj*dxHdx)

      if (Update .gt. 1) then
         H0scal = ydx / dxHdx

*        Modifiy the LM updates so that  H  is multiplied by  H0scal.
*        The arrays  ytdx(1:nQNmod)  and  dxtHdx(1:nQNmod)  hold the
*        inverse values of previous  ydx  and  dxHdx.

         dxHdx  = dxHdx  / H0scal ! Saved for the LC case
         dxHdxI = dxHdxI * H0scal
         call dscal ( nnH, H0scal, H0, 1 )

         if (nQNmod .gt. 0) then
            do k = 1, nQNmod
               ytdx  (k) = ytdx  (k)*H0scal
               dxtHdx(k) = dxtHdx(k)*H0scal
            end do
         end if
      end if

      if (nQNmod .ge. mQNmod) then
*        ---------------------------------------------------------------
*        Insufficient space for storing the new (Hdx,y).
*        Reset H0 to be the diagonal of the current H.
*        Discard any updates accumulated so far.
*        ---------------------------------------------------------------
         call s9LMH0( Htype, nnH, mQNmod, nQNmod,
     &        H0pre, H0, gdif, Hxdif, ytdx, dxtHdx )

*        If H0 was retained, include the latest update.

         if (Htype .ne. HUnit) then
            do i = 1, nnH
               H0(i) = H0(i) - Hdx(i)*Hdx(i)*dxHdxI + y(i)*y(i)*ydxI
            end do
         end if

         nQNmod = 0

      else
*        ---------------------------------------------------------------
*        Space remains. Store Hdx and y.
*        ---------------------------------------------------------------
         nQNmod = nQNmod + 1
         call dcopy ( nnH, Hdx, 1, Hxdif(1,nQNmod), 1 )
         call dcopy ( nnH, y  , 1,  gdif(1,nQNmod), 1 )
         ytdx  (nQNmod) = ydxI
         dxtHdx(nQNmod) = dxHdxI
         Htype  = HNorml

      end if

      end ! of s9LMup

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine s9LMHx( nnH, x, Hx, mQNmod, nQNmod,
     &     H0, gdif, Hxdif, ytdx, dxtHdx )

      implicit
     &     none
      integer
     &     nnH, mQNmod, nQNmod
      double precision
     &     Hx(nnH), x(nnH), H0(nnH), ytdx(mQNmod), dxtHdx(mQNmod),
     &     gdif(nnH,mQNmod), Hxdif(nnH,mQNmod)

*     ==================================================================
*     s9LMHx does the work for s8Hx.
*
*     19 Jul 1995: First version of s9LMHx
*     11 Sep 1999: Current version.
*     ==================================================================
      integer
     &     k
      double precision
     &     c1, c2, ddot
*     ------------------------------------------------------------------
      call dcopy ( nnH,  x, 1, Hx, 1 )
      call ddscl ( nnH, H0, 1, Hx, 1 )

      if (nQNmod .gt. 0) then
         do k  = 1, nQNmod
            c1 = - ddot ( nnH, Hxdif(1,k), 1, x, 1 )*dxtHdx(k)
            call daxpy ( nnH, c1, Hxdif(1,k), 1, Hx, 1 )

            c2 =   ddot ( nnH,  gdif(1,k), 1, x, 1 )*  ytdx(k)
            call daxpy ( nnH, c2, gdif (1,k), 1, Hx, 1 )
         end do
      end if

      end ! of s9LMHx

