*     ------------------------------------------------------------------
*     File springi.f for SNOPT                            (Unix Version)
*
*     This is a main program to generate an optimal control problem
*     of arbitrary size and solve it by calling SNOPT as a subroutine.
*
*     The problem is identical to Spring.f except that the constraints
*     are infeasible.
*
*     The problem size depends on a parameter T.  There are
*     2T constraints and 3T + 2 variables, as well as bounds
*     on the variables.  The first T constraints are quadratic in
*     T + 1 variables, and the objective function is quadratic in
*     T + 1 other variables.
*
*     The control problem models a spring, mass and damper system.
*     It is of the form
*
*   --------------------------------------------------------------------
*   | minimize    1/2 sum x(t)**2   (t = 0 to T)                       |
*   |                                                                  |
*   | subject to                                                       |
*   |     y(t+1)  =  y(t)  -  0.01 y(t)**2  -  0.004 x(t)  +  0.2 u(t) |
*   |                                                                  |
*   |     x(t+1)  =  x(t)  +  0.2  y(t),                               |
*   |                                                                  |
*   |     y(t)   >=  -1,     -0.2  <=  u(t)  <=  0.2,                  |
*   |                                                                  |
*   |                (all for t = 0 to T-1)                            |
*   | and                                                              |
*   |     y(0)    =   0,      y(T)  =  0,       x(0) = 10.             |
*   --------------------------------------------------------------------
*
*     For large enough T (e.g. T >= 90), the optimal objective value
*     is about 1186.382.
*     
*     This model with T = 100 was used as test problem 5.11 in
*     B. A. Murtagh and M. A. Saunders (1982), A projected Lagrangian
*     algorithm and its implementation for sparse nonlinear constraints,
*     Mathematical Programming Study 16, 84--117.
*     
*     14 Nov 1994: First version of springi.f, derived from manne.f.
*     24 Jul 1997: Updated for SNOPT 5.2.
*     ------------------------------------------------------------------
      program            spring

      implicit           double precision (a-h,o-z)

      parameter        ( maxT   = 2000,
     $                   maxm   = 2*maxT,
     $                   maxn   = 3*maxT + 2,
     $                   maxne  = 7*maxT,
     $                   nName  = 1 )

      character*8        ProbNm      , Names(nName)
      integer            T
      integer            ha(maxne)   , hs(maxm+maxn)
      integer            ka(maxn+1)
      double precision   a(maxne)    , bl(maxm+maxn), bu(maxm+maxn),
     $                   x(maxm+maxn), pi(maxm)     , rc(maxm+maxn)

*     USER workspace (none required)

      parameter          (  lenru = 1)
      double precision   ru(lenru)
      parameter          (  leniu = 1) 
      integer            iu(leniu)
      parameter          (  lencu = 1) 
      character*8        cu(lencu)

*     SNOPT workspace

      parameter          (  lenrw = 1000000)
      double precision   rw(lenrw)
      parameter          (  leniw =  500000) 
      integer            iw(leniw)
      parameter          (  lencw =     500) 
      character*8        cw(lencw)

      logical            byname
      character*20       lfile
      external           SprCon, SprObj
*     ------------------------------------------------------------------
*     Specify some of the SNOPT files.
*     iSpecs  is the Specs file   (0 if none).
*     iPrint  is the Print file   (0 if none).
*     iSumm   is the Summary file (0 if none).
*     nout    is an output file used here by the main program.

      iSpecs = 4
      iPrint = 15
      iSumm  = 6
      nout   = 6

      byname = .true.
 
      if ( byname ) then

*        Unix and DOS systems.  Open the Specs and print files.

         lfile = 'springi.spc'
         open( iSpecs, file=lfile, status='OLD',     err=800 )

         lfile = 'springi.out'
         open( iPrint, file=lfile, status='UNKNOWN', err=800 )
      end if

*     ------------------------------------------------------------------
*     Set options to their default values.
*     ------------------------------------------------------------------
      call snInit( iPrint, iSumm,
     $             cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Read a Specs file.  This must include "Nonlinear constraints  T"
*     for some integer T.
*     ------------------------------------------------------------------
      call snSpec( iSpecs, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )

      if (inform .ge. 2) then
         write(nout, *) 'iSpecs > 0 but no Specs file found'
         stop
      end if

*     ------------------------------------------------------------------
*     The following call fetches nnCon, which defines T, the number of
*     nonlinear constraints.
*     It is specified at runtime in the SPECS file.
*     ------------------------------------------------------------------
      call sngeti( 'Nonlinear constraints', nnCon,
     $             inform, cw, lencw, iw, leniw, rw, lenrw )
      T     = nnCon
      if (T .le. 1  .or.  T .gt. maxm/2) then
         write(nout, *) 'Invalid no. of Nonlinear constraints:', T
         stop
      end if

*     Write T into the problem name.

      write(ProbNm, '(i8)') T
      if      (T .lt.   100) then
         ProbNm(1:6) = 'Spring'
      else if (T .lt.  1000) then
         ProbNm(1:5) = 'Sprng'
      else if (T .lt. 10000) then
         ProbNm(1:4) = 'Spri'
      else
         ProbNm(1:3) = 'Spr'
      end if

      write(nout, *) 'Spring inf. optimal control problem.    T =', T

*     ------------------------------------------------------------------
*     Generate an T-period problem.
*     ------------------------------------------------------------------
      call spdata( T, maxm, maxn, maxne, inform,
     $             m, n, ne, nnCon, nnObj, nnJac,
     $             a, ha, ka, bl, bu, hs, x, pi )

      if (inform .ge. 1) then
         write(nout, *) 'Not enough storage to generate a problem ',
     $                  'with  Nonlinear constraints =', T
         stop
      end if             

*     ------------------------------------------------------------------
*     Specify options that were not set in the Specs file.
*     i1 and i2 may refer to the Print and Summary file respectively.
*     Setting them to 0 suppresses printing.
*     ------------------------------------------------------------------
      maxS   = T
      itnlim = T * 10
      i1     = 0
      i2     = 0
      call snseti( 'Superbasics Limit ', maxS  , i1, i2, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )
      call snseti( 'Iterations        ', itnlim, i1, i2, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Go for it, using a Cold start.
*     iobj   = 0 means there is no linear objective row in a(*).
*     Objadd = 0.0 means there is no constant to be added to the
*            objective.
*     hs     need not be set if a basis file is to be input.
*            Otherwise, each hs(1:n) should be 0, 1, 2, 3, 4, or 5.
*            The values are used by the Crash procedure
*            to choose an initial basis B.
*            If hs(j) = 0 or 1, column j is eligible for B.
*            If hs(j) = 2, column j is initially superbasic (not in B).
*            If hs(j) = 3, column j is eligible for B and is given
*                          preference over columns with hs(j) = 0 or 1.
*            If hs(j) = 4 or 5, column j is initially nonbasic.
*     ------------------------------------------------------------------
      iObj   = 0
      ObjAdd = 0.0d+0

      call snopt ( 'Cold', m, n, ne, nName,
     $             nnCon, nnObj, nnJac,
     $             iObj, ObjAdd, ProbNm,
     $             SprCon, SprObj,
     $             a, ha, ka, bl, bu, Names,
     $             hs, x, pi, rc, 
     $             inform, mincw, miniw, minrw,
     $             nS, ninf, sinf, obj,
     $             cu, lencu, iu, leniu, ru, lenru, 
     $             cw, lencw, iw, leniw, rw, lenrw )

      if (inform .eq. 42 .or. inform .eq. 43 .or. inform .eq. 44) then
         write(nout, *) ' '
         write(nout, *) 'Estimate of required lencw:', mincw
         write(nout, *) 'Estimate of required leniw:', miniw
         write(nout, *) 'Estimate of required lenrw:', minrw
         stop
      end if

      write(nout, *) ' '
      write(nout, *) 'SNOPT finished.'
      write(nout, *) 'inform =', inform
      write(nout, *) 'nInf   =', nInf
      write(nout, *) 'sInf   =', sInf
      write(nout, *) 'Obj    =', Obj
      if (inform .ge. 20) go to 900
      stop

*     ------------------------------------------------------------------
*     Error exit.
*     ------------------------------------------------------------------
  800 write(nout, 4000) 'Error while opening file', lfile
      stop

  900 write(nout, *) ' '
      write(nout, *) 'STOPPING because of error condition'
      stop

 4000 format(/  a, 2x, a  )

*     end of main program for springi problem
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine spdata( T, maxm, maxn, maxne, inform,
     $                   m, n, ne, nnCon, nnObj, nnJac,
     $                   a, ha, ka, bl, bu, hs, x, pi )

      implicit           double precision (a-h,o-z)
      integer            T
      integer            ha(maxne)   , hs(maxn+maxm)
      integer            ka(maxn+1)
      double precision   a(maxne)    , bl(maxn+maxm), bu(maxn+maxm),
     $                   x(maxn+maxm), pi(maxm)

*     ------------------------------------------------------------------
*     spdata generates data for the "Spring" optimal control problem.
*     The constraints take the form
*              c(x) + A*x - s = 0,
*     where the Jacobian for c(x) + Ax is stored in a(*), and any
*     terms coming from c(x) are in the TOP LEFT-HAND CORNER of a(*),
*     with dimensions  nnCon x nnJac.
*     Note that the right-hand side is zero.
*     s is a set of slack variables whose bounds contain any constants
*     that might have formed a right-hand side.
*
*     The objective function is
*             f(x) + d'x
*     where d would be row iobj of A (but there is no such row in
*     this example).  f(x) involves only the FIRST nnObj variables.
*   
*     On entry,
*     T       is the number of time periods.
*     maxm, maxn, maxne are upper limits on m, n, ne.
*
*     On exit,
*     inform  is 0 if there is enough storage, 1 otherwise.
*     m       is the number of nonlinear and linear constraints.
*     n       is the number of variables.
*     ne      is the number of nonzeros in a(*).
*     nnCon   is the number of nonlinear constraints (they come first).
*     nnObj   is the number of nonlinear objective variables.
*     nnJac   is the number of nonlinear Jacobian variables.
*     a       is the constraint matrix (Jacobian), stored column-wise.
*     ha      is the list of row indices for each nonzero in a(*).
*     ka      is a set of pointers to the beginning of each column of a.
*     bl      is the lower bounds on x and s.
*     bu      is the upper bounds on x and s.
*     hs(1:n) is a set of initial states for each x  (0,1,2,3,4,5).
*     x (1:n) is a set of initial values for x.
*     pi(1:m) is a set of initial values for the dual variables pi.
*
*     14 Nov 1994: First version of spdata.
*     ------------------------------------------------------------------
      parameter      ( zero   = 0.0d+0,   one    = 1.0d+0 )
      parameter      ( bplus  = 1.0d+20,  dummy  = 0.111111d+0 )

*     T defines the dimension of the problem.

      m      = T*2
      n      = T*3 + 2
      nb     = n   + m
      nnCon  = T
      nnObj  = T*2 + 2  ! y(0:T) and x(0:T)
      nnJac  = T   + 1  ! y(0:T)
      ne     = T*7

*     Check if there is enough storage.

      inform = 0
      if (m      .gt. maxm ) inform = 1
      if (n      .gt. maxn ) inform = 1
      if (ne     .gt. maxne) inform = 1
      if (inform .gt.   0  ) return

*     ------------------------------------------------------------------
*     Generate columns for y(t), t = 0 to T.
*     The first T rows are nonlinear, and the next T are linear.
*     The Jacobian is T x (T+1) upper bidiagonal.
*     We generate the sparsity pattern here.
*     We put in dummy numerical values for the nonlinear gradients.
*     The true non-constant values are computed by funcon.
*     ------------------------------------------------------------------
      j      = 0   ! counts the variables
      ne     = 0   ! counts the Jacobian and linear constraint entries

      do 100, k = 0, T
         j      =   j  + 1
         ka(j)  =   ne + 1
         bl(j)  = - one
         bu(j)  =   bplus
         x (j)  = - one
         hs(j)  =   0      ! Make the y(t) nonbasic.

*        There are two Jacobian nonzeros per column,
*        except in the first and last column.

         if (k .gt. 0) then    !  Aij = 1
            ne     =   ne + 1
            ha(ne) =   k
            a(ne)  =   one
         end if

         if (k .lt. T) then    !  Aij = .02y - 1  (nonlinear)
            ne     =   ne + 1
            ha(ne) =   k  + 1
            a(ne)  =   dummy
         end if

*        Below the Jacobian the linear constraints are diagonal.

         if (k .lt. T) then
            ne     =   ne + 1
            ha(ne) =   T  + k + 1
            a(ne)  = - 0.2d+0
         end if
  100 continue

*     ------------------------------------------------------------------
*     Generate columns for x(t), t = 0 to T.
*     They form 0.004*I in the first T rows,
*     and an upper-bidiagonal in the last T rows.
*     ------------------------------------------------------------------
      do 200, k = 0, T
         j      =   j  + 1
         ka(j)  =   ne + 1
         bl(j)  = - bplus
         bu(j)  =   bplus
         x (j)  =   zero
         hs(j)  =   3     ! Make the x(t) basic.

*        Part of 0.004*I.

         if (k .lt. T) then
            ne     =   ne + 1
            ha(ne) =   k  + 1
            a(ne)  =   0.004d+0
         end if

*        The bidiagonal parts have two entries
*        except in the first and last columns.

         if (k .gt. 0) then    !  Aij = 1
            ne     =   ne + 1
            ha(ne) =   T  + k
            a(ne)  =   one
         end if

         if (k .lt. T) then    !  Aij = - 1
            ne     =   ne + 1
            ha(ne) =   T  + k + 1
            a(ne)  = - one
         end if
  200 continue

*     ------------------------------------------------------------------
*     Generate columns for u(t), t = 0 to T-1.
*     They form -0.2I in the first T rows.
*     ------------------------------------------------------------------
      do 300, k = 0, T - 1
         j      =   j  + 1
         ka(j)  =   ne + 1
         bl(j)  = - 0.2d+0
         bu(j)  =   0.2d+0
         x (j)  =   zero
         hs(j)  =   3     ! Make the u(t) basic.

         ne     =   ne + 1
         ha(ne) =   k  + 1
         a(ne)  = - 0.2d+0
  300 continue

*     ka(*) has one extra element.
*     Some of the variables are fixed.

      ka(n+1) = ne + 1
      bl(1)   = zero      ! y(0) = 0
      bu(1)   = zero
      bl(T+1) = zero      ! y(T) = 0
      bu(T+1) = zero
      bl(T+2) = 10.0d+0   ! x(0) = 10
      bu(T+2) = 10.0d+0

*     Added 11/9/96 to make springii infeasible

      bl(2*T+3) = -10.0d+0   ! x(T) = -10
      bu(2*T+3) = -10.0d+0
      
*     ------------------------------------------------------------------
*     Set bounds on the slacks.
*     We don't need to set initial values and states for slacks
*     (assuming SNOPT does a cold start).
*     ------------------------------------------------------------------
      do 500, j = n + 1, nb
         bl(j)  = zero
         bu(j)  = zero
  500 continue

*     Initialize pi.
*     SNOPT requires only pi(1:nnCon) to be initialized.
*     We initialize all of pi just in case.

      do 600,  i = 1, T
         pi(i)   =   zero
         pi(T+i) =   zero
  600 continue

*     end of spdata
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine SprObj( mode, n, x, f, g, nState,
     $                   cu, lencu, iu, leniu, ru, lenru )

      implicit           double precision (a-h,o-z)

      double precision   x(n), g(n)

      character*8        cu(lencu)
      integer            iu(leniu)
      double precision   ru(lenru)  

*     ------------------------------------------------------------------
*     This is funobj for problem Springi  (an optimal control problem).
*     ------------------------------------------------------------------
      integer            T
      parameter        ( zero = 0.0d+0 )

      T      = (n - 2)/2
      f      = zero
      jy     = 0
      jx     = T + 1

      do 50, k = 0, T
         jy    = jy + 1
         jx    = jx + 1
         u     = x(jx)
         f     = f  +  u**2
         g(jy) = zero
         g(jx) = u
   50 continue

      f = f / 2.0d+0

*     end of objective for Springi
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine SprCon( mode, m, n, njac, x, f, g, nState, 
     $                   cu, lencu, iu, leniu, ru, lenru )

      implicit           double precision (a-h,o-z)
      double precision   x(n), f(m), g(njac)

      character*8        cu(lencu)
      integer            iu(leniu)
      double precision   ru(lenru)  

*     ------------------------------------------------------------------
*     This is funcon for problem Springi  (Optimal Control Problem).
*     The Jacobian is upper bidiagonal,
*     and only the diagonal terms are nonlinear.
*     The constant 1's in the Jacobian are not regenerated here.
*     ------------------------------------------------------------------
      integer            T
      parameter        ( one = 1.0d+0 )

      T     = n - 1
      jy    =     0    ! Counts y(t) variables
      jg    =   - 1    ! Counts nonlinear Jacobian elements

      do 150, i = 1, T
         jy     = jy + 1
         jg     = jg + 2
         yt     = x(jy)
         ytp1   = x(jy + 1)
         f(i)   = 0.01d+0 * yt**2  +  (ytp1  -  yt)
         g(jg)  = 0.02d+0 * yt               -  one
*--      g(jg+1)= one      ! Constant element set by spdata.
  150 continue

*     end of constraints for Springi
      end

