*     ------------------------------------------------------------------
*     File sqmain.f (Unix version)
*     This is a simple example of a call to subroutine SQOPT, which is
*     part of the SNOPT package.
*
*     04 Oct 1994: First   version.
*     18 May 1998: Current version.
*     ------------------------------------------------------------------
      program            sqmain
      implicit           double precision (a-h,o-z)

      parameter        ( maxm   = 10000,
     $                   maxn   = 15000,
     $                   maxne  = 30000 )

      character*8        Prob
      character*8        Names(maxn+maxm)
      integer*4          ha(maxne) , hElast(maxn+maxm), hs(maxn+maxm)
      integer            ka(maxn+1)
      double precision   a(maxne)  , bl(maxn+maxm)    , bu(maxn+maxm) , 
     $                   c(maxn)   , xs(maxn+maxm)    ,
     $                   pi(maxm)  , rc(maxn+maxm)

*     SQOPT workspace

      parameter          (  lenrw = 10000)
      double precision   rw(lenrw)
      parameter          (  leniw =  5000) 
      integer            iw(leniw)
      parameter          (  lencw =   500) 
      character*8        cw(lencw)

      logical            byname
      character*20       lfile
      external           userHx

*     ------------------------------------------------------------------
*     Specify some of the SQOPT files.
*     iSpecs  is the Specs file   (0 if none).
*     iPrint  is the Print file   (0 if none).
*     iSumm   is the Summary file (0 if none).
*
*     nout    is an output file used here by main.

      iSpecs = 4
      iPrint = 15
      iSumm  = 6
      nout   = 6

      byname = .true.
 
      if ( byname ) then

*        Unix and DOS systems.  Open the Specs and print files.

         lfile = 'sqmain.spc'
         open( iSpecs, file=lfile, status='OLD',     err=800 )

         lfile = 'sqmain.out'
         open( iPrint, file=lfile, status='UNKNOWN', err=800 )
      end if

*     ------------------------------------------------------------------
*     Define the problem.
*     (1) Compute l, u, and A so that the constraints are ranges of the
*         form  l <= Ax <= u.
*         Store l and u in bl(n+1:n+m) and bu(n+1:n+m).
*
*     (2) Set up the constants ObjAdd and c so that the explicit
*         objective is 
*             ObjAdd + c'*x + half*x'*H*x
*         If necessary, include an additional linear objective terms
*         as row iObj of A. 
*     ------------------------------------------------------------------
      call sqdat1( maxm, maxn, maxne, 
     $             m, n, ne, nName, lenc, ncolH,
     $             iObj, ObjAdd, Prob,
     $             a, ha, ka, bl, bu, c,
     $             Names, xs )

*     ------------------------------------------------------------------
*     Set the initial value and status of each variable.
*     For want of something better to do, make the variables xs(1:n)
*     temporarily fixed at their current values. 
*     The crash can set the rest.
*     ------------------------------------------------------------------
      do 100, j = 1, n
         hs(j) = 0
  100 continue

*     ------------------------------------------------------------------
*     Fix the column variables to be non-elastic and the row  variables 
*     to be elastic.
*     ------------------------------------------------------------------
      do 110, j = 1, n
         hElast(j) = 0
  110 continue
 
      do 120, j = n+1, n+m
         hElast(j) = 3
  120 continue

*     ------------------------------------------------------------------
*     First,  sqInit MUST be called to initialize optional parameters 
*     to their default values.
*     ------------------------------------------------------------------
      call sqInit( iPrint, iSumm,
     $             cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Read a Specs file (Optional).
*     ------------------------------------------------------------------
      call sqSpec( iSpecs, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )

      if (inform .ge. 2) then
         write(nout, *) 'iSpecs > 0 but no Specs file found'
         go to 990
      end if

*     ------------------------------------------------------------------
*     Specify options that were not set in the Specs file.
*     iP and iS may refer to the Print and Summary file respectively.
*     Setting them to 0 suppresses printing.
*     ------------------------------------------------------------------
      maxS   = ncolH + 1
      itnlim = 40
      iP     =  0
      iS     =  0
      call sqseti( 'Superbasics Limit ', maxS  , iP, iS, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )
      call sqseti( 'Iterations        ', itnlim, iP, iS, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Go for it, using a Cold start.
*     hs     need not be set if a basis file is to be input.
*            Otherwise, each hs(1:n) should be 0, 1, 2, 3, 4, or 5.
*            The values are used by the Crash procedure m2crsh
*            to choose an initial basis B.
*            If hs(j) = 0 or 1, column j is eligible for B.
*            If hs(j) = 2, column j is initially superbasic (not in B).
*            If hs(j) = 3, column j is eligible for B and is given
*                          preference over columns with hs(j) = 0 or 1.
*            If hs(j) = 4 or 5, column j is initially nonbasic.
*     ------------------------------------------------------------------
      call sqopt ( 'Cold', userHx, m, 
     $             n, ne, nName, lenc, ncolH,
     $             iObj, ObjAdd, Prob,
     $             a, ha, ka, bl, bu, c, Names, 
     $             hElast, hs, xs, pi, rc, 
     $             inform, mincw, miniw, minrw,
     $             nS, ninf, sinf, Obj,
     $             cw, lencw, iw, leniw, rw, lenrw, 
     $             cw, lencw, iw, leniw, rw, lenrw )

      if (inform .eq. 42 .or. inform .eq. 43 .or. inform .eq. 44) then
         write(nout, *) ' '
         write(nout, *) 'Estimate of required lencw:', mincw
         write(nout, *) 'Estimate of required leniw:', miniw
         write(nout, *) 'Estimate of required lenrw:', minrw
         go to 990
      end if

      write(nout, *) ' '
      write(nout, *) 'sqopt finished.'
      write(nout, *) 'inform =', inform
      write(nout, *) 'ninf   =', ninf
      write(nout, *) 'sinf   =', sinf
      write(nout, *) 'obj    =', obj
      if (inform .ge. 20) go to 910

*     ------------------------------------------------------------------
*     Solve the same problem defined with different data.
*     Since the dimensions are slightly different, use a Cold start.
*     ------------------------------------------------------------------
      call sqdat2( maxm, maxn, maxne, 
     $             m, n, ne, nName, lenc, ncolH,
     $             iObj, ObjAdd, Prob,
     $             a, ha, ka, bl, bu, c,
     $             Names, xs )

*     ------------------------------------------------------------------
*     Set the initial value and status of each variable.
*     For fun, we use the hs values from the previous run.
*     The crash can set the rest.
*     ------------------------------------------------------------------
      do 200, j = 1, n
         hs(j) = 0
  200 continue
*     ------------------------------------------------------------------
*     Fix the column variables to be non-elastic and the row  variables 
*     to be elastic.
*     ------------------------------------------------------------------
      do 210, j = 1, n
         hElast(j) = 0
  210 continue

      do 220, j = n+1, n+m
         hElast(j) = 3
  220 continue

      call sqopt ( 'Cold', userHx, m, 
     $             n, ne, nName, lenc, ncolH,
     $             iObj, ObjAdd, Prob,
     $             a, ha, ka, bl, bu, c, Names, 
     $             hElast, hs, xs, pi, rc, 
     $             inform, mincw, miniw, minrw,
     $             nS, nInf, sInf, Obj,
     $             cw, lencw, iw, leniw, rw, lenrw, 
     $             cw, lencw, iw, leniw, rw, lenrw )

      if (inform .eq. 42 .or. inform .eq. 43 .or. inform .eq. 44) then
         write(nout, *) ' '
         write(nout, *) 'Estimate of required lencw:', mincw
         write(nout, *) 'Estimate of required leniw:', miniw
         write(nout, *) 'Estimate of required lenrw:', minrw
         go to 990
      end if

      write(nout, *) ' '
      write(nout, *) 'sqopt finished.'
      write(nout, *) 'inform =', inform
      write(nout, *) 'nInf   =', nInf
      write(nout, *) 'sInf   =', sInf
      write(nout, *) 'obj    =', obj
      if (inform .ge. 20) go to 910

*     ------------------------------------------------------------------
*     Alter some options and call sqopt again, testing the Warm start.
*     The following illustrates the use of sqset, sqseti and sqsetr
*     to set specific options.  We could ensure that all unspecified
*     options take default values by first calling
*     sqset ( 'Defaults', ... ).
*     Beware that certain parameters would then need to be redefined.
*     ------------------------------------------------------------------
      write(nout, *) ' '
      write(nout, *) 'Alter options and test Warm start:'

      inform = 0
      itnlim = 500
      call sqset ( 'Defaults          ',         iPrint, iSumm, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )
      call sqset ( '                  ',         iPrint, iSumm, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )
      call sqset ( 'Print  level     0',         iPrint, iSumm, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )
      call sqset ( 'Scale option     0',         iPrint, iSumm, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )
      call sqseti( 'Iterations        ', itnlim, iPrint, iSumm, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )

      if (inform .gt. 0) then
         write(nout, *) 'NOTE: Some of the options were not recognized'
      end if

*     ------------------------------------------------------------------
*     Test the Warm start.
*     hs(*) specifies a complete basis from the previous call.
*     A Warm start uses hs(*) directly, without calling Crash.
*     
*     Warm starts are normally used after sqopt has solved a
*     problem with the SAME DIMENSIONS but perhaps altered data.
*     Here we have not altered the data, so very few iterations
*     should be required.
*     ------------------------------------------------------------------
      call sqopt ( 'Warm', userHx, m, 
     $             n, ne, nName, lenc, ncolH,
     $             iObj, ObjAdd, Prob,
     $             a, ha, ka, bl, bu, c, Names, 
     $             hElast, hs, xs, pi, rc, 
     $             inform, mincw, miniw, minrw,
     $             nS, ninf, sinf, Obj,
     $             cw, lencw, iw, leniw, rw, lenrw, 
     $             cw, lencw, iw, leniw, rw, lenrw )

      write(nout, *) ' '
      write(nout, *) 'sqopt finished again.'
      write(nout, *) 'inform =', inform
      write(nout, *) 'obj    =', obj
      if (inform .ge. 20) go to 910
      stop

*     ------------------------------------------------------------------
*     Error exit.
*     ------------------------------------------------------------------
  800 write(nout, 4000) 'Error while opening file', lfile
      stop

  910 write(nout, *) ' '
      write(nout, *) 'STOPPING because of error condition'

  990 stop

 4000 format(/  a, 2x, a  )

*     end of main program to test subroutine sqopt
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine userHx( ncolH, x, Hx, nState, 
     $                   cu, lencu, iu, leniu, ru, lenru )

      implicit           double precision (a-h,o-z)
      double precision   x(ncolH), Hx(ncolH)

      character*8        cu(lencu)
      integer            iu(leniu)
      double precision   ru(lenru)

*     ==================================================================
*     This is the user-defined Hessian-vector product for the
*     sqopt example program.
*
*          ( 2  0  0  0  0  0  0 )
*          ( 0  2  0  0  0  0  0 )
*          ( 0  0  2  2  0  0  0 )
*     H =  ( 0  0  2  2  0  0  0 )
*          ( 0  0  0  0  2  0  0 )
*          ( 0  0  0  0  0  0  0 )
*          ( 0  0  0  0  0  0  0 )  
*
*     ==================================================================
      parameter         (two   = 2.0d+0)
      parameter         (nOut  = 6)
*     ------------------------------------------------------------------

*     First entry.  Print something on the standard output.

      if (nState .eq. 1) then
         if (nOut .gt. 0) write(nOut, 1000) ncolH
      end if

*     -------------
*     Normal entry.
*     -------------
      Hx(1) = two*x(1)
      Hx(2) = two*x(2)
      Hx(3) = two*x(3) + two*x(4)
      Hx(4) = two*x(4) + two*x(3)
      Hx(5) = two*x(5)
*     Hx(6) = two*x(6) + two*x(7)
*     Hx(7) = two*x(7) + two*x(6)
         
*     ------------
*     Final entry.
*     ------------
      if (nState .ge. 2) then
         if (nOut .gt. 0) write(nOut, 2000)
      end if
      return

 1000 format(// ' This is problem  sqmain.   ncolH =', i4/)
 2000 format(// ' Finished         sqmain.')

*     end of userHx
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine sqdat1( maxm, maxn, maxne, 
     $                   m, n, ne, nName, lenc, ncolH,
     $                   iObj, ObjAdd, Prob,
     $                   a, ha, ka, bl, bu, c,
     $                   Names, xs )

      implicit           double precision (a-h,o-z)
      character*8        Prob
      character*8        Names(maxn+maxm)
      integer            ha(maxne)
      integer            ka(maxn+1)

      double precision   a(maxne)
      double precision   bl(maxn+maxm), bu(maxn+maxm), xs(maxn+maxm)
      double precision   c(maxn)

*     ==================================================================
*     sqdat1   defines the first problem for the SQOPT documentation.
*
*     (1) Compute l, u, and A so that the constraints are ranges of the
*         form  l <= Ax <= u.
*         Store l and u in bl(n+1:n+m) and bu(n+1:n+m).
*
*     (2) Set up the constants ObjAdd and c so that the explicit
*         objective is 
*             ObjAdd + c'*x + half*x'*H*x
*         If necessary, include an additional linear objective terms
*         as row iObj of A. 
*     ==================================================================
*
*     Name the Problem.

      Prob   = 'sqdata 1'

*     ------------------------------------------------------------------
*     Assign the constraint nonzeros to a, column by column.
*     ha(i) gives the row index of element a(i).
*     ka(j) gives the index in a of the start of column j.
*     ------------------------------------------------------------------
*
*     2000     ( 1.0    1.0    1.0    1.0    1.0    1.0    1.0  )   2000
*     -inf     ( 0.15   0.04   0.02   0.04   0.02   0.01   0.03 )     60
*     -inf     ( 0.03   0.05   0.08   0.02   0.06   0.01   0.0  )    100
*     -inf le  ( 0.02   0.04   0.01   0.02   0.02   0.0    0.0  ) le  40
*     -inf     ( 0.02   0.03   0.0    0.0    0.01   0.0    0.0  )     30
*     1500     ( 0.70   0.75   0.80   0.75   0.80   0.97   0.0  )    inf
*      250     ( 0.02   0.06   0.08   0.12   0.02   0.01   0.97 )    300 
*     ------------------------------------------------------------------

      n       = 7
      m       = 7      ! Does not include an objective row
      ne      = 41     ! n*m for a dense A.

      lenc    = 7
      ncolH   = 5
      nName   = 1

      BigBnd  =  1.0d+20

*     Column 1.

      ka( 1) =  1

      ha( 1) =  1
      ha( 2) =  2 
      ha( 3) =  3 
      ha( 4) =  4 
      ha( 5) =  5 
      ha( 6) =  6 
      ha( 7) =  7

      a( 1)  = 1.0d+0
      a( 2)  = 0.15d+0
      a( 3)  = 0.03d+0
      a( 4)  = 0.02d+0
      a( 5)  = 0.02d+0
      a( 6)  = 0.70d+0
      a( 7)  = 0.02d+0
      
*     Column 2.

      ka( 2) =  8

      ha( 8) =  1
      ha( 9) =  2
      ha(10) =  3
      ha(11) =  4 
      ha(12) =  5 
      ha(13) =  6 
      ha(14) =  7 

      a( 8)  = 1.0d+0
      a( 9)  = 0.04d+0
      a(10)  = 0.05d+0
      a(11)  = 0.04d+0
      a(12)  = 0.03d+0
      a(13)  = 0.75d+0
      a(14)  = 0.06d+0

*     Column 3.

      ka( 3) = 15

      ha(15) =  1
      ha(16) =  2
      ha(17) =  3
      ha(18) =  4 
      ha(19) =  6 
      ha(20) =  7

      a(15)  = 1.0d+0
      a(16)  = 0.02d+0
      a(17)  = 0.08d+0
      a(18)  = 0.01d+0
      a(19)  = 0.80d+0
      a(20)  = 0.08d+0

*     Column 4.

      ka( 4) = 21

      ha(21) =  1
      ha(22) =  2
      ha(23) =  3
      ha(24) =  4 
      ha(25) =  6 
      ha(26) =  7 

      a(21) = 1.0d+0
      a(22) = 0.04d+0
      a(23) = 0.02d+0
      a(24) = 0.02d+0
      a(25) = 0.75d+0
      a(26) = 0.12d+0

*     Column 5.

      ka( 5) = 27

      ha(27) =  1
      ha(28) =  2
      ha(29) =  3
      ha(30) =  4 
      ha(31) =  5 
      ha(32) =  6 
      ha(33) =  7 

      a(27)  = 1.0d+0
      a(28)  = 0.02d+0
      a(29)  = 0.06d+0
      a(30)  = 0.02d+0
      a(31)  = 0.01d+0
      a(32)  = 0.80d+0
      a(33)  = 0.02d+0

*     Column 6.

      ka(6)  = 34

      ha(34) =  1
      ha(35) =  2
      ha(36) =  3
      ha(37) =  6 
      ha(38) =  7 

      a(34)  = 1.0d+0
      a(35)  = 0.01d+0
      a(36)  = 0.01d+0
      a(37)  = 0.97d+0
      a(38)  = 0.01d+0

*     Column 7. 

      ka(7)  =  39

      ha(39) =  1
      ha(40) =  2
      ha(41) =  7 

      a(39)  =  1.0d+0
      a(40)  =  0.03d+0
      a(41)  =  0.97d+0

*     Don't forget to finish off  ka.
*     This is crucial.

      ka( 8) =  ne + 1

*     ------------------------------------------------------------------
*     Set the upper and lower bounds on the variables
*     ------------------------------------------------------------------
      bl(1) =    0.0d+0
      bu(1) =  200.0d+0

      bl(2) =    0.0d+0
      bu(2) = 2500.0d+0

      bl(3) =  400.0d+0
      bu(3) =  800.0d+0

      bl(4) =  100.0d+0
      bu(4) =  700.0d+0

      bl(5) =    0.0d+0
      bu(5) = 1500.0d+0

      bl(6) =    0.0d+0
      bu(6) =    BigBnd

      bl(7) =    0.0d+0
      bu(7) =    BigBnd

*     ------------------------------------------------------------------
*     Set the upper and lower bounds on  Ax.
*     ------------------------------------------------------------------
      bl(n+1) =    2000d+0
      bu(n+1) =    2000d+0

      bl(n+2)  = - BigBnd
      bu(n+2)  =   60.0d+0

      bl(n+3)  = - BigBnd
      bu(n+3)  =  100.0d+0 

      bl(n+4)  = - BigBnd
      bu(n+4)  =   40.0d+0 

      bl(n+5)  = - BigBnd 
      bu(n+5)  =   30   

      bl(n+6)  =   1500d+0    
      bu(n+6)  =   BigBnd  

      bl(n+7)  =    250.0d+0
      bu(n+7)  =    300d+0   

*     ------------------------------------------------------------------
*     Set the objective terms.
*     The objective linear term is explicit.
*     ------------------------------------------------------------------
      c(1)     = - 200.0d+0
      c(2)     = -2000.0d+0
      c(3)     = -2000.0d+0
      c(4)     = -2000.0d+0
      c(5)     = -2000.0d+0
      c(6)     =   400.0d+0
      c(7)     =   400.0d+0

      iObj     = 0
      ObjAdd   = 0.0d+0

*     ------------------------------------------------------------------
*     Set the initial estimate of the solution.
*     ------------------------------------------------------------------
      xs(1) =   0.0d+0
      xs(2) = 700.0d+0
      xs(3) = 500.0d+0
      xs(4) = 400.0d+0
      xs(5) =   0.0d+0
      xs(6) = 300.0d+0
      xs(7) = 200.0d+0

*     end of sqdat1
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine sqdat2( maxm, maxn, maxne, 
     $                   m, n, ne, nName, lenc, ncolH,
     $                   iObj, ObjAdd, Prob,
     $                   a, ha, ka, bl, bu, c,
     $                   Names, xs )

      implicit           double precision (a-h,o-z)
      character*8        Prob
      character*8        Names(maxn+maxm)
      integer            ha(maxne)
      integer            ka(maxn+1)

      double precision   a(maxne)
      double precision   bl(maxn+maxm), bu(maxn+maxm), xs(maxn+maxm)
      double precision   c(maxn)

*     ------------------------------------------------------------------
*     sqdat2   defines the second problem for the SQOPT documentation.
*
*     (1) Compute l, u, and A so that the constraints are ranges of the
*         form  l <= Ax <= u.
*         Store l and u in bl(n+1:n+m) and bu(n+1:n+m).
*
*     (2) Set up the constants ObjAdd and c so that the explicit
*         objective is 
*             ObjAdd + c'*x + half*x'*H*x
*         If necessary, include an additional linear objective terms
*         as row iObj of A. 
*     ------------------------------------------------------------------
*
*     Name the Problem.

      Prob   = 'sqdata 2'

*     ------------------------------------------------------------------
*     Assign the constraint nonzeros to a, column by column.
*     ha(i) gives the row index of element a(i).
*     ka(j) gives the index in a of the start of column j.
*     ------------------------------------------------------------------
*
*     2000     ( 1.0    1.0    1.0    1.0    1.0    1.0    1.0  )   2000
*     -inf     ( 0.15   0.04   0.02   0.04   0.02   0.01   0.03 )     60
*     -inf     ( 0.03   0.05   0.08   0.02   0.06   0.01   0.0  )    100
*     -inf le  ( 0.02   0.04   0.01   0.02   0.02   0.0    0.0  ) le  40
*     -inf     ( 0.02   0.03   0.0    0.0    0.01   0.0    0.0  )     30
*     1500     ( 0.70   0.75   0.80   0.75   0.80   0.97   0.0  )    inf
*      250     ( 0.02   0.06   0.08   0.12   0.02   0.01   0.97 )    300 
*     -inf     ( -200  -2000  -2000  -2000  -2000   400    400  )    inf 
*     ------------------------------------------------------------------

      n       =  7
      m       =  8              ! A now includes a free objective row
      ne      = 48
      nName   =  1

      lenc    =  0
      ncolH   =  5

      BigBnd  =  1.0d+20

*     Column 1.

      ka( 1) =  1

      ha( 1) =  1
      ha( 2) =  2 
      ha( 3) =  3 
      ha( 4) =  4 
      ha( 5) =  5 
      ha( 6) =  6 
      ha( 7) =  7
      ha( 8) =  8

      a( 1)  =  1.0d+0
      a( 2)  =  0.15d+0
      a( 3)  =  0.03d+0
      a( 4)  =  0.02d+0
      a( 5)  =  0.02d+0
      a( 6)  =  0.70d+0
      a( 7)  =  0.02d+0
      a( 8)  = -200.0d+0
      
*     Column 2.

      ka( 2) =  9

      ha( 9) =  1
      ha(10) =  2
      ha(11) =  3
      ha(12) =  4 
      ha(13) =  5 
      ha(14) =  6 
      ha(15) =  7 
      ha(16) =  8 

      a( 9)  =  1.0d+0
      a(10)  =  0.04d+0
      a(11)  =  0.05d+0
      a(12)  =  0.04d+0
      a(13)  =  0.03d+0
      a(14)  =  0.75d+0
      a(15)  =  0.06d+0
      a(16)  = -2000.0d+0

*     Column 3.

      ka( 3) = 17

      ha(17) =  1
      ha(18) =  2
      ha(19) =  3
      ha(20) =  4 
      ha(21) =  6 
      ha(22) =  7
      ha(23) =  8

      a(17)  = 1.0d+0
      a(18)  = 0.02d+0
      a(19)  = 0.08d+0
      a(20)  = 0.01d+0
      a(21)  = 0.80d+0
      a(22)  = 0.08d+0
      a(23)  = -2000.0d+0

*     Column 4.

      ka( 4) = 24

      ha(24) =  1
      ha(25) =  2
      ha(26) =  3
      ha(27) =  4 
      ha(28) =  6 
      ha(29) =  7 
      ha(30) =  8 

      a(24) = 1.0d+0
      a(25) = 0.04d+0
      a(26) = 0.02d+0
      a(27) = 0.02d+0
      a(28) = 0.75d+0
      a(29) = 0.12d+0
      a(30) = -2000.0d+0

*     Column 5.

      ka( 5) = 31

      ha(31) =  1
      ha(32) =  2
      ha(33) =  3
      ha(34) =  4 
      ha(35) =  5 
      ha(36) =  6 
      ha(37) =  7 
      ha(38) =  8 

      a(31)  = 1.0d+0
      a(32)  = 0.02d+0
      a(33)  = 0.06d+0
      a(34)  = 0.02d+0
      a(35)  = 0.01d+0
      a(36)  = 0.80d+0
      a(37)  = 0.02d+0
      a(38) = -2000.0d+0

*     Column 6.

      ka(6)  = 39

      ha(39) =  1
      ha(40) =  2
      ha(41) =  3
      ha(42) =  6 
      ha(43) =  7 
      ha(44) =  8 

      a(39)  = 1.0d+0
      a(40)  = 0.01d+0
      a(41)  = 0.01d+0
      a(42)  = 0.97d+0
      a(43)  = 0.01d+0
      a(44)  = 400.0d+0

*     Column 7. 

      ka(7)  =  45

      ha(45) =  1
      ha(46) =  2
      ha(47) =  7 
      ha(48) =  8 

      a(45)  =  1.0d+0
      a(46)  =  0.03d+0
      a(47)  =  0.97d+0
      a(48) =   400.0d+0

*     Don't forget to finish off  ka.
*     This is crucial.

      ka( 8) =  ne + 1

*     ------------------------------------------------------------------
*     Set the upper and lower bounds on the variables
*     ------------------------------------------------------------------
      bl(1) =    0.0d+0
      bu(1) =  200.0d+0

      bl(2) =    0.0d+0
      bu(2) = 2500.0d+0

      bl(3) =  400.0d+0
      bu(3) =  800.0d+0

      bl(4) =  100.0d+0
      bu(4) =  700.0d+0

      bl(5) =    0.0d+0
      bu(5) = 1500.0d+0

      bl(6) =    0.0d+0
      bu(6) =    BigBnd

      bl(7) =    0.0d+0
      bu(7) =    BigBnd

*     ------------------------------------------------------------------
*     Set the upper and lower bounds on  Ax.
*     ------------------------------------------------------------------
      bl(n+1) =  2000d+0
      bu(n+1) =  2000d+0

      bl(n+2) = - BigBnd
      bu(n+2) =  60.0d+0

      bl(n+3) = - BigBnd
      bu(n+3) = 100.0d+0 

      bl(n+4) = - BigBnd
      bu(n+4) =  40.0d+0 

      bl(n+5) = - BigBnd 
      bu(n+5) =  30.0d+0   

      bl(n+6) =  1500d+0    
      bu(n+6) =   BigBnd  

      bl(n+7) =   250.0d+0
      bu(n+7) =   300.0d+0   

*     The last row is the free objective row

      bl(n+8)  = - BigBnd
      bu(n+8)  =   BigBnd

*     ------------------------------------------------------------------
*     Set the objective terms.
*     There is no explicit objective term
*     ------------------------------------------------------------------
      iObj     = 8
      ObjAdd   = 0.0d+0

*     ------------------------------------------------------------------
*     Set the initial estimate of the solution.
*     ------------------------------------------------------------------
      xs(1) =   0.0d+0
      xs(2) = 700.0d+0
      xs(3) = 500.0d+0
      xs(4) = 400.0d+0
      xs(5) =   0.0d+0
      xs(6) = 300.0d+0
      xs(7) = 200.0d+0

*     end of sqdat2
      end
