*     ------------------------------------------------------------------
*     File t2banana.f  (Unix version)
*     Illustrates using SNOPT on an unconstrained problem.
*
*     15 May 1998: First   version.
*     15 May 1998: Current version.
*     ------------------------------------------------------------------
      program            t2main
      implicit           double precision (a-h,o-z)

      parameter        ( maxm   = 1000,
     $                   maxn   = 1000,
     $                   maxne  = 3000,
     $                   nName  = 1 )

      character*8        Prob
      character*8        Names(nName)
      integer*4          ha(maxne) , hs(maxn+maxm)
      integer            ka(maxn+1)
      double precision   a(maxne)  , bl(maxn+maxm), bu(maxn+maxm),
     $                   x(maxn+maxm), pi(maxm)   , rc(maxn+maxm)

*     SNOPT workspace

      parameter          (  lenrw = 20000)
      double precision   rw(lenrw)
      parameter          (  leniw = 10000) 
      integer            iw(leniw)
      parameter          (  lencw =   500) 
      character*8        cw(lencw)
      logical            byname
      character*20       lfile
      external           dummy, t2obj

*     ------------------------------------------------------------------
*     Specify some of the SNOPT files.
*     iSpecs  is the Specs file   (0 if none).
*     iPrint  is the Print file   (0 if none).
*     iSumm   is the Summary file (0 if none).
*
*     nout    is an output file used here by t2banana.

      iSpecs = 4
      iPrint = 15
      iSumm  = 6
      nout   = 6

      byname = .true.
 
      if ( byname ) then

*        Unix and DOS systems.  Open the Specs and print files.

         lfile = 't2banana.spc'
         open( iSpecs, file=lfile, status='OLD',     err=900 )

         lfile = 't2banana.out'
         open( iPrint, file=lfile, status='UNKNOWN', err=900 )
      end if

*     ------------------------------------------------------------------
*     First,  snInit MUST be called to initialize optional parameters 
*     to their default values.
*     ------------------------------------------------------------------
      call snInit( iPrint, iSumm,
     $             cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Read a Specs file (Optional).
*     ------------------------------------------------------------------
      call snSpec( iSpecs, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )

      if (inform .ge. 2) then
         write(nout, *) 'iSpecs > 0 but no Specs file found'
         go to 800
      end if

*     Set up the data structure for the sparse Jacobian.
*     Assign dummy values for the nonlinear elements. 

      call t2data( maxm, maxn, maxne, inform, 
     $             m, n, ne, nnCon, nnObj, nnJac,
     $             iObj, ObjAdd, Prob,
     $             a, ha, ka, bl, bu, hs, x, pi )

*     ------------------------------------------------------------------
*     Specify any options not set in the Specs file.
*     i1 and i2 may refer to the Print and Summary file respectively.
*     Setting them to 0 suppresses printing.
*     ------------------------------------------------------------------
      itnlim = 250
      i1     =   0
      i2     =   0
      call snseti( 'Iterations        ', itnlim, i1, i2, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Go for it, using a Cold start.
*     hs     need not be set if a basis file is to be input.
*            Otherwise, each hs(1:n) should be 0, 1, 2, 3, 4, or 5.
*            The values are used by the Crash procedure m2crsh
*            to choose an initial basis B.
*            If hs(j) = 0 or 1, column j is eligible for B.
*            If hs(j) = 2, column j is initially superbasic (not in B).
*            If hs(j) = 3, column j is eligible for B and is given
*                          preference over columns with hs(j) = 0 or 1.
*            If hs(j) = 4 or 5, column j is initially nonbasic.
*     ------------------------------------------------------------------
      call snopt ( 'Cold', m, n, ne, nName,
     $             nnCon, nnObj, nnJac,
     $             iObj, ObjAdd, Prob,
     $             dummy, t2obj,
     $             a, ha, ka, bl, bu, Names,
     $             hs, x, pi, rc, 
     $             inform, mincw, miniw, minrw,
     $             nS, nInf, sInf, Obj,
     $             cw, lencw, iw, leniw, rw, lenrw,
     $             cw, lencw, iw, leniw, rw, lenrw )

      write(nout, *) ' '

      if (inform .eq. 42 .or. inform .eq. 43 .or. inform .eq. 44) then
         write(nout, *) 'Estimate of required lencw:', mincw
         write(nout, *) 'Estimate of required leniw:', miniw
         write(nout, *) 'Estimate of required lenrw:', minrw
      else
         write(nout, *) 'snopt finished.'
         write(nout, *) 'inform =', inform
         write(nout, *) 'nInf   =', nInf
         write(nout, *) 'sInf   =', sInf
         write(nout, *) 'Obj    =', Obj
      end if

      if (inform .ge. 20) then
         write(nout, *) ' '
         write(nout, *) 'STOPPING because of error condition'
      end if

  800 if ( byname ) then
         close(iSpecs)
         close(iPrint)
      end if

      stop

*     ------------------------------------------------------------------
*     File Error.
*     ------------------------------------------------------------------
  900 write(nout, 4000) 'Error while opening file', lfile
      stop

 4000 format(/  a, 2x, a  )

*     end of t2main.
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine t2obj ( mode, n, x, f, g, nState,
     $                   cw, lencw, iw, leniw, rw, lenrw )

      implicit           double precision (a-h,o-z)
      double precision   x(n), g(n)

      character*8        cw(lencw)
      integer            iw(leniw)
      double precision   rw(lenrw)

*     ------------------------------------------------------------------
*     This is funobj for problem t2banana
*     (Rosenbrock's banana function).
*     f(x) = 100(x2 - x1**2)**2 + (1 - x1)**2.
*     ------------------------------------------------------------------
      x1     =   x(1)
      x2     =   x(2)
      t1     =   x2 - x1**2
      t2     =   1.0d+0 - x1
      f      =   100.0d+0 * t1**2  +  t2**2
      if (mode .eq. 0) return

      g(1)   = - 400.0d+0 * t1*x1  -  2.0d+0 * t2
      g(2)   =   200.0d+0 * t1

*     end of t2obj  (funobj for t2banana)
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine dummy ( mode, nnCon, nnJac, neJac, x, fCon, gCon,
     $                   nState, cu, lencu, iu, leniu, ru, lenru )

      implicit           double precision(a-h,o-z)
      double precision   x(nnJac), fCon(nnCon), gCon(neJac)

      character*8        cu(lencu)
      integer            iu(leniu)
      double precision   ru(lenru)

*     ==================================================================
*     Problem t2banana.
*     No nonlinear constraints.
*     ==================================================================

*     Relax

*     end of dummy constraints for t2banana
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine t2data( maxm, maxn, maxne, inform, 
     $                   m, n, ne, nnCon, nnObj, nnJac,
     $                   iObj, ObjAdd, Prob,
     $                   a, ha, ka, bl, bu, hs, x, pi )

      implicit           double precision (a-h,o-z)
      character*8        Prob
      integer            ha(maxne)   , hs(maxn+maxm)
      integer            ka(maxn+1)
      double precision   a(maxne)    , bl(maxn+maxm), bu(maxn+maxm),
     $                   x(maxn+maxm), pi(maxm)

*     ------------------------------------------------------------------
*     Define the problem.
*     (1) Compute l, u, and A so that the constraints are ranges of the
*         form  l <= Ax <= u.
*         Store l and u in bl(n+1:n+m) and bu(n+1:n+m).
*
*     (2) Set up the constants ObjAdd and iObj so that the explicit
*         objective is 
*             ObjAdd + (row iObj of A)'*x + f(x) 
*
*     On entry,
*     maxm, maxn, maxne are upper limits on m, n, ne.
*
*     On exit,
*     inform  is 0 if there is enough storage, 1 otherwise.
*     m       is the number of nonlinear and linear constraints.
*     n       is the number of variables.
*     ne      is the number of nonzeros in a(*).
*     nnCon   is the number of nonlinear constraints (they come first).
*     nnObj   is the number of nonlinear objective variables.
*     nnJac   is the number of nonlinear Jacobian variables.
*     a       is the constraint matrix (Jacobian), stored column-wise.
*     ha      is the list of row indices for each nonzero in a(*).
*     ka      is a set of pointers to the beginning of each column of a.
*     bl      is the lower bounds on x and s.
*     bu      is the upper bounds on x and s.
*     hs(1:n) is a set of initial states for each x  (0,1,2,3,4,5).
*     x (1:n) is a set of initial values for x.
*     pi(1:m) is a set of initial values for the dual variables pi.
*     ------------------------------------------------------------------
      parameter         (bplus   =  1.0d+21)
      parameter         (zero    =  0.0d+0,   one    = 1.0d+0)
      parameter         (ten     = 10.0d+0)

*     Name the Problem.

      Prob = 't2Banana'

      n     =  2
      m     =  1
      ne    =  1                ! Dummy row

*     Check if there is enough storage.

      inform = 0
      if (m      .gt. maxm ) inform = 1
      if (n      .gt. maxn ) inform = 1
      if (ne     .gt. maxne) inform = 1
      if (inform .gt.   0  ) return

      nnCon   = 0
      nnJac   = 0
      nnObj   = n

*     ==================================================================
*     SNOPT requires at least one row in a, so we have to
*     provide a dummy linear constraint. 
*
*     Define a (sparse) free row that has just one element. 
*     Set this element to be zero.
*     Note that ka(j) = ka(j+1) for all columns j that have no entries.
*     ------------------------------------------------------------------
      ka( 1) =  1
      ha( 1) =  1
      a ( 1) =  zero

      do 100, j = 2, n
         ka(j) = 2
  100 continue
      ka(n+1) =  ne + 1

*     Make the linear row a free row.

      bl(n+1) =  -bplus
      bu(n+1) =   bplus

*     ------------------------------------------------------------------
*     Set the upper and lower bounds on the variables
*     ------------------------------------------------------------------
      do 200, j = 1, n
         bl(j) =  -ten
         bu(j) =   ten
  200 continue

      bu(1) = 5.0d+0

*     No linear objective term for this problem.

      iObj    = 0
      ObjAdd  = zero

*     ------------------------------------------------------------------
*     Initialize x, hs and pi.
*     Set the initial value and status of each variable.
*     For want of something better to do, make the variables x(1:n)
*     temporarily fixed at their current values. 
*     The crash can set the rest.
*     ------------------------------------------------------------------
      x(1) = - 1.2d+0
      x(2) =   1.0d+0

      do 400, j = 1, n
         hs(j)  = 0
  400 continue

      do 500, i = 1, m
         pi(i)  = zero
  500 continue

*     end of t2data.
      end

