*     ------------------------------------------------------------------
*     File t4manne.f (Unix version)
*     Illustrates using SNOPT on a constrained problem.
*     It generates the problem called MANNE on Pages 98-108 of the
*     MINOS 5.1 User's Guide, then asks snopt to solve it.
*
*     16 May 1998: First   version.
*     16 May 1998: Current version.
*     ------------------------------------------------------------------
      program            t4main

      implicit           double precision (a-h,o-z)

      parameter        ( maxm   = 1000,
     $                   maxn   = 1500,
     $                   maxne  = 3000,
     $                   nName  = 1 )

      character*8        ProbNm      , Names(nName)
      integer            ha(maxne)   , hs(maxm+maxn)
      integer            ka(maxn+1)
      double precision   a(maxne)    , bl(maxm+maxn), bu(maxm+maxn),
     $                   x(maxm+maxn), pi(maxm)     , rc(maxm+maxn)

*     SNOPT workspace

      parameter          (  lenrw = 1000000)
      double precision   rw(lenrw)
      parameter          (  leniw =  500000) 
      integer            iw(leniw)
      parameter          (  lencw =     500) 
      character*8        cw(lencw)
      logical            byname
      character*20       lfile
      external           ManCon, ManObj

*     ------------------------------------------------------------------
*     Specify some of the SNOPT files.
*     iSpecs  is the Specs file   (0 if none).
*     iPrint  is the Print file   (0 if none).
*     iSumm   is the Summary file (0 if none).
*     nout    is an output file used here by sntest.

      iSpecs = 4   ! equivalenced to t4manne.spc
      iPrint = 15  ! equivalenced to t4manne.out
      iSumm  = 6   ! summary file goes to standard output...
      nout   = 6   ! ... as do messages from this program.

      byname = .true.
 
      if ( byname ) then

*        Unix and DOS systems.  Open the Specs and print files.

         lfile = 't4manne.spc'
         open( iSpecs, file=lfile, status='OLD',     err=800 )

         lfile = 't4manne.out'
         open( iPrint, file=lfile, status='UNKNOWN', err=800 )
      end if

*     ------------------------------------------------------------------
*     Set options to default values.
*     ------------------------------------------------------------------
      call snInit( iPrint, iSumm,
     $             cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Read a Specs file.  This must include "Nonlinear constraints  T"
*     for some integer T.
*     ------------------------------------------------------------------
      call snSpec( iSpecs, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )

      if (inform .ge. 2) then
         write(nout, *) 'iSpecs > 0 but no Specs file found'
         go to 910
      end if

*     ------------------------------------------------------------------
*     The following assignment allows access to nnCon,
*     which defines T, the number of nonlinear constraints.
*     It is specified at runtime in the SPECS file.
*     ------------------------------------------------------------------
      call sngeti( 'Nonlinear constraints', nnCon,
     $             inform, cw, lencw, iw, leniw, rw, lenrw )
      nT        = nnCon
      if (nT .le. 1  .or.  nT .gt. maxm/2) then
         write(nout, *) 'Invalid  nT  specified:', nT
         go to 910
      end if

*     Write nT into the problem name.

      write(probnm, '(i8)') nT
      if      (nT .lt.  1000) then
         probnm(1:5) = 'Manne'
      else if (nT .lt. 10000) then
         probnm(1:4) = 'Mann'
      else
         probnm(1:3) = 'Man'
      end if

      write(nout, *) 'Problem MANNE.    T =', nT

*     ------------------------------------------------------------------
*     Generate an nT-period problem.
*     ------------------------------------------------------------------
      call t4data( nT, maxm, maxn, maxne, inform,
     $             m, n, ne, nnCon, nnObj, nnJac,
     $             a, ha, ka, bl, bu, hs, x, pi )

      if (inform .ge. 1) then
         write(nout, *) 'Not enough storage to generate a problem ',
     $                  'with  nT =', nT
         go to 910
      end if             

*     ------------------------------------------------------------------
*     Specify options that were not set in the Specs file.
*     i1 and i2 may refer to the Print and Summary file respectively.
*     Setting them to 0 suppresses printing.
*     ------------------------------------------------------------------
      maxS   =  2*nT
      itnlim = 40*nT
      i1     = 0
      i2     = 0
      call snseti( 'Superbasics Limit ', maxS  , i1, i2, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )
      call snseti( 'Iterations        ', itnlim, i1, i2, inform,
     $             cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Go for it, using a Cold start.
*     iObj   = 0 means there is no linear objective row in a(*).
*     ObjAdd = 0.0 means there is no constant to be added to the
*            objective.
*     hs     need not be set if a basis file is to be input.
*            Otherwise, each hs(1:n) should be 0, 1, 2, 3, 4, or 5.
*            The values are used by the Crash procedure m2crsh
*            to choose an initial basis B.
*            If hs(j) = 0 or 1, column j is eligible for B.
*            If hs(j) = 2, column j is initially superbasic (not in B).
*            If hs(j) = 3, column j is eligible for B and is given
*                          preference over columns with hs(j) = 0 or 1.
*            If hs(j) = 4 or 5, column j is initially nonbasic.
*
*     SNOPT is called with iw and rw used for USER workspace.
*     This allows access to SNOPT variables in  ManCon and ManObj.
*     ------------------------------------------------------------------
      iObj   = 0
      ObjAdd = 0.0d+0

      call snopt ( 'Cold', m, n, ne, nName,
     $             nnCon, nnObj, nnJac,
     $             iObj, ObjAdd, ProbNm,
     $             ManCon, ManObj,
     $             a, ha, ka, bl, bu, Names,
     $             hs, x, pi, rc, 
     $             inform, mincw, miniw, minrw,
     $             nS, nInf, sInf, Obj,
     $             cw, lencw, iw, leniw, rw, lenrw, 
     $             cw, lencw, iw, leniw, rw, lenrw )

      if (inform .eq. 42 .or. inform .eq. 43 .or. inform .eq. 44) then
         write(nout, *) ' '
         write(nout, *) 'Estimate of required lencw:', mincw
         write(nout, *) 'Estimate of required leniw:', miniw
         write(nout, *) 'Estimate of required lenrw:', minrw
         go to 910
      end if

      write(nout, *) ' '
      write(nout, *) 'snopt finished.'
      write(nout, *) 'inform =', inform
      write(nout, *) 'ninf   =', ninf
      write(nout, *) 'sInf   =', sInf
      write(nout, *) 'Obj    =', Obj
      if (inform .ge. 20) go to 910
      stop

*     ------------------------------------------------------------------
*     Error exit.
*     ------------------------------------------------------------------
  800 write(nout, 4000) 'Error while opening file', lfile
      stop

  910 write(nout, *) ' '
      write(nout, *) 'STOPPING because of error condition'

  990 stop

 4000 format(/  a, 2x, a  )

*     end of main program to test subroutine snopt
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine t4data( nT, maxm, maxn, maxne, inform,
     $                   m, n, ne, nnCon, nnObj, nnJac,
     $                   a, ha, ka, bl, bu, hs, x, pi )

      implicit           double precision (a-h,o-z)
      integer            ha(maxne), hs(maxn+maxm)
      integer            ka(maxn+1)
      double precision   a(maxne)    , bl(maxn+maxm), bu(maxn+maxm),
     $                   x(maxn+maxm), pi(maxm)

*     ------------------------------------------------------------------
*     t4data  generates data for the test problem t4manne
*     (called problem MANNE in the SNOPT 4.3 User's Guide).
*     The constraints take the form
*              f(x) + A*x - s = 0,
*     where the Jacobian for f(x) + Ax is stored in a(*), and any
*     terms coming from f(x) are in the TOP LEFT-HAND CORNER of a(*),
*     with dimensions  nnCon x nnJac.
*     Note that the right-hand side is zero.
*     s is a set of slack variables whose bounds contain any constants
*     that might have formed a right-hand side.
*
*     The objective function is
*             f(x) + c'x
*     where c would be row iObj of A (but there is no such row in
*     this example).  f(x) involves only the FIRST nnObj variables.
*   
*     On entry,
*     nT      is T, the number of time periods.
*     maxm, maxn, maxne are upper limits on m, n, ne.
*
*     On exit,
*     inform  is 0 if there is enough storage, 1 otherwise.
*     m       is the number of nonlinear and linear constraints.
*     n       is the number of variables.
*     ne      is the number of nonzeros in a(*).
*     nnCon   is the number of nonlinear constraints (they come first).
*     nnObj   is the number of nonlinear objective variables.
*     nnJac   is the number of nonlinear Jacobian variables.
*     a       is the constraint matrix (Jacobian), stored column-wise.
*     ha      is the list of row indices for each nonzero in a(*).
*     ka      is a set of pointers to the beginning of each column of a.
*     bl      is the lower bounds on x and s.
*     bu      is the upper bounds on x and s.
*     hs(1:n) is a set of initial states for each x (0,1,2,3,4,5).
*     x(1:n)  is a set of initial values for x.
*     pi(1:m) is a set of initial values for the dual variables pi.
*
*     09 Jul 1992: No need to initialize x and hs for the slacks.
*     15 Oct 1993: pi is now an output parameter.  (Should have been
*                  all along.)
*     ------------------------------------------------------------------
      parameter      ( zero   = 0.0d+0,   one    = 1.0d+0,
     $                 dummy  = 0.1d+0,   growth = .03d+0,
     $                 bplus  = 1.0d+20,  bminus = - bplus )

*     nT defines the dimension of the problem.

      m      = nT*2
      n      = nT*3
      nnCon  = nT
      nnObj  = nT*2
      nnJac  = nT
      ne     = nT*6 - 1
      T      = nT

*     Check if there is enough storage.

      inform = 0
      if (m      .gt. maxm ) inform = 1
      if (n      .gt. maxn ) inform = 1
      if (ne     .gt. maxne) inform = 1
      if (inform .gt.   0  ) return

*     Generate columns for Capital (Kt, t = 1 to nT).
*     The first nT rows are nonlinear, and the next nT are linear.
*     The Jacobian is an nT x nT diagonal.
*     We generate the sparsity pattern here.
*     We put in dummy numerical values of 0.1 for the gradients.
*     Real values for the gradients are computed by funcon.

      ne     = 0
      do 100  k = 1, nT

*        There is one Jacobian nonzero per column.

         ne     = ne + 1
         ka(k)  = ne
         ha(ne) = k
         a(ne)  = dummy

*        The linear constraints form an upper bidiagonal pattern.

         if (k .gt. 1) then
            ne     = ne + 1
            ha(ne) = nT + k - 1
            a(ne)  = one
         end if

         ne     = ne + 1
         ha(ne) = nT + k
         a(ne)  = - one
  100 continue

*     The last nonzero is special.

      a(ne)  = growth

*     Generate columns for Consumption (Ct for t = 1 to nT).
*     They form -I in the first nT rows.
*     jC and jI are base indices for the Ct and It variables.

      jC    = nT
      jI    = nT*2

      do 200 k = 1, nT
         ne       = ne + 1
         ka(jC+k) = ne
         ha(ne)   = k
         a(ne)    = - one
  200 continue

*     Generate columns for Investment (It for t = 1 to nT).
*     They form -I in the first nT rows and -I in the last nT rows.

      do 300 k = 1, nT
         ne       = ne + 1
         ka(jI+k) = ne
         ha(ne)   = k
         a(ne)    = - one
         ne       = ne + 1
         a(ne)    = - one
         ha(ne)   = nT + k
  300 continue

*     ka(*) has one extra element.

      ka(n+1) = ne + 1

*     Set lower and upper bounds for Kt, Ct, It.
*     Also initial values and initial states for all variables.
*     The Jacobian variables are the most important.
*     We make them all superbasic.
*     The others are ok nonbasic.
*     For test purposes, we want the initial x to be infeasible
*     with respect to the linear constraints.
*     Try setting the last Kapital too high.


      do 400  k = 1, nT
         bl(   k) = 3.05d+0
         bu(   k) = bplus
         bl(jC+k) = 0.95d+0
         bu(jC+k) = bplus
         bl(jI+k) = 0.05d+0
         bu(jI+k) = bplus

         x(   k)  = 3.0d+0 + (k - 1)/10.0d+0
         x(jC+k)  = bl(jC+k)
         x(jI+k)  = bl(jI+k)

*-->     hs(   k) = 2
         hs(   k) = 0

         hs(jC+k) = 0
         hs(jI+k) = 0

         if (k .eq. nT) then
            x(k)  = 1.0d+3
            hs(k) = 2
         end if
  400 continue

*     The first Capital is fixed.
*     The last three Investments are bounded.
*     Fudge them to be the normal ones for T = 10.

      scale       = T / 10.0d+0
      bu(1)       = bl(1)
      x(1)        = bl(1)
      hs(1)       = 0
      bu(jI+nT-2) = 0.112d+0 * scale
      bu(jI+nT-1) = 0.114d+0 * scale
      bu(jI+nT  ) = 0.116d+0 * scale

*     Set bounds on the slacks.
*     The nT nonlinear (Money)    rows are >=.
*     The nT    linear (Capacity) rows are <=.
*     We no longer need to set initial values and states for slacks
*     (assuming SNOPT does a cold start).

      jM     = n
      jY     = n + nT

      do 500    k = 1, nT
         bl(jM+k) = zero
         bu(jM+k) = bplus
         bl(jY+k) = bminus
         bu(jY+k) = zero

*-       x (jM+k) = zero
*-       x (jY+k) = zero
*-       hs(jM+k) = 0
*-       hs(jY+k) = 0
  500 continue

*     The last Money and Capacity rows have a Range.

      bu(jM+nT) =   10.0d+0
      bl(jY+nT) = - 20.0d+0

*     Initialize pi.
*     SNOPT requires only pi(1:nnCon) to be initialized.
*     We initialize all of pi just in case.

      do 600 i = 1, nT
         pi(i)    = - one
         pi(nT+i) = + one
  600 continue

*     end of t4data
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine ManObj( mode, n, x, f, g, nState,
     $                   cw, lencw, iw, leniw, rw, lenrw )

      implicit           double precision (a-h,o-z)
      double precision   x(n), g(n)

      character*8        cw(lencw)
      integer            iw(leniw)
      double precision   rw(lenrw)  

*     ------------------------------------------------------------------
*     This is funobj for problem t4manne.
*
*     The data bt(*) is computed by funcon  on its first entry.
*
*     For test purposes, we look at    Derivative level
*     and sometimes pretend that we don't know the first
*     three elements of the gradient.
*
*     12-Nov-93: Changed from Maximize to Minimize to steer around bugs.
*     13-Jan-95: Reverted to Maximize.
*     ==================================================================
      intrinsic          log
      logical            gknown
      parameter        ( zero = 0.0d+0 )
      common    /manne / b,at(100),bt(100)
*     ------------------------------------------------------------------
      lvlDer    = iw( 70) ! = 0, 1, 2 or 3, the derivative level

      gknown = lvlDer .eq. 1  .or.  lvlDer .eq. 3
      nT     = n/2
      f      = zero

      do 50 j = 1, nT
         xcon = x(nT+j)
         f    = f  +  bt(j) * log(xcon)
*Min     f    = f  -  bt(j) * log(xcon)
         if (mode .gt. 0) then
            g(j) = zero
            if (gknown                ) g(nT+j) = + bt(j) / xcon
*           if (gknown  .or.  j .gt. 3) g(nT+j) = + bt(j) / xcon
*Min        if (gknown  .or.  j .gt. 3) g(nT+j) = - bt(j) / xcon
         end if
   50 continue

*     end of ManObj (funobj for t4manne)
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine ManCon( mode, m, n, njac, x, f, g, nState, 
     $                   cw, lencw, iw, leniw, rw, lenrw )

      implicit           double precision (a-h,o-z)
      double precision   x(n), f(m), g(njac)

      character*8        cw(lencw)
      integer            iw(leniw)
      double precision   rw(lenrw)  

*     ==================================================================
*     This is funcon for problem t4manne.
*
*     For test purposes, we look at    Derivative level
*     and sometimes pretend that we don't know the first
*     three elements of the gradient.
*     ==================================================================
      common    /manne / b,at(100),bt(100)
*     ------------------------------------------------------------------
      logical            gknown
      parameter        ( one = 1.0d+0 )

      iPrint    = iw( 12) ! Print file
      lvlDer    = iw( 70) ! = 0, 1, 2 or 3, the derivative level

      gknown = lvlDer .ge. 2
      nT     = n

*     ---------------------------------------
*     First entry.  Define b, at(*) and bt(*) 
*     for this and all subsequent entries.
*     ---------------------------------------
      if (nstate .eq. 1) then
         grow   = 0.03d+0
         beta   = 0.95d+0
         xk0    = 3.0d+0
         xc0    = 0.95d+0
         xi0    = 0.05d+0
         b      = 0.25d+0
         if (iPrint .gt. 0) write(iPrint, 1000) nt, b
      
         a      = (xc0 + xi0) / xk0**b
         gfac   = (one + grow)**(one - b)
         at(1)  = a*gfac
         bt(1)  = beta

         do 10 j  = 2, nT
            at(j) = at(j-1)*gfac
            bt(j) = bt(j-1)*beta
   10    continue

         bt(nT) = bt(nT) / (one - beta)
      end if

*     -------------
*     Normal entry.
*     -------------
      do 150 j = 1, nT
         xkap  = x(j)
         f(j)  = at(j) * xkap**b
         if (mode .gt. 0) then
            if (gknown                ) g(j) = b*f(j) / xkap
*           if (gknown  .or.  j .gt. 3) g(j) = b*f(j) / xkap
         end if
  150 continue

*     ------------
*     Final entry.
*     ------------
      if (nstate .ge. 2) then
         if (iPrint .gt. 0) write(iPrint, 2000) (f(j), j = 1, nT)
      end if
      return

 1000 format(// ' This is problem  t4manne.   nT =', i4, '   b =', f8.3)
 2000 format(// ' Final nonlinear function values' / (5f12.5))

*     end of ManCon (funcon for t4manne)
      end

