*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
*     File  npopt.f
*
*     npopt    npwrap   npHess   npload   npprnt   
*
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine npopt ( n, nclin, ncnln, ldA, ldcJ, ldH,
     &     A, bl, bu, funcon, funobj,
     &     inform, majIts, iState,
     &     cCon, cJac, cMul, Objf, grad, Hess, x,
     &     iw, leniw, rw, lenrw )

      implicit
     &     none
      external
     &     funcon, funobj
      integer
     &     n, nclin, ncnln, ldA, ldcJ, ldH, inform, majIts,
     &     leniw, lenrw, iState(n+nclin+ncnln), iw(leniw)
      double precision
     &     Objf, A(ldA,*), bl(n+nclin+ncnln), bu(n+nclin+ncnln),
     &     cCon(*), cJac(ldcJ,*), cMul(n+nclin+ncnln), grad(n),
     &     Hess(ldH,*), x(n), rw(lenrw)

*     ==================================================================
*     npopt    solves the nonlinear programming problem
*
*            minimize                   f(x)
*
*                                    (      x  )
*            subject to    bl  .le.  (    A*x  )  .le.  bu
*                                    ( cCon(x) )
*
*     where  f(x)  is a smooth scalar function,  A  is a constant matrix
*     and  cCon(x)  is a vector of smooth nonlinear functions.
*     The feasible region is defined by a mixture of linear and 
*     nonlinear equality or inequality constraints on  x.
*
*     The calling sequence of NPOPT and the user-defined functions
*     funcon and funobj are identical to those of the dense code NPSOL
*     (see the User's Guide for NPSOL (Version 4.0): a Fortran Package
*     for Nonlinear Programming, Systems Optimization Laboratory Report
*     SOL 86-2, Department of Operations Research, Stanford University,
*     1986.)
*
*     The dimensions of the problem are...
*
*     n        the number of variables (dimension of  x),
*
*     nclin    the number of linear constraints (rows of the matrix  A),
*
*     ncnln    the number of nonlinear constraints (dimension of  c(x)),
*
*     NPOPT is maintained by Philip E. Gill, 
*     Dept of Mathematics, University of California, San Diego.
*     
*     LUSOL is maintained by Michael A. Saunders,
*     Systems Optimization Laboratory,
*     Dept of Management Science & Engineering, Stanford University.
*
*     22 Mar 1997: First version of npopt.
*     17 Jul 1997: First thread-safe version.
*     11 Oct 1998: Added facility to combine funobj and funcon.
*     11 Nov 2000: Current version of npopt.
*     ==================================================================
      integer
     &     Htype, i, iObj, iError, iPrint, iSumm, lbl, lbu, lencw,
     &     lenR, lFcon, lGcon, lGobj, lhs, lindJ, lJcol, llocJ, lNames,
     &     lpi, lprSol, lprSav, lrc, lvlSrt, lvlHes, lx, m, maxcw,
     &     maxiw, maxR, maxrw, maxS, mincw, miniw, minrw, mProb, mQNmod, 
     &     nb, nCon, ne, nInf, neG, nlocJ, nGobj, nMajor, nName, nnCol,
     &     nnCon0, nnCon, nnJac, nnObj, nS, Start, tolfac, tolupd
      double precision
     &     dnrm1s, Fobj, ObjAdd, ObjTru, sInf, xNorm
      external
     &     snprnt, sqprnt, npwrap
*     ------------------------------------------------------------------
      integer            FM
      parameter         (FM     = 1) 
      integer            Cold,       Warm 
      parameter         (Cold   = 0, Warm   = 2)
      integer            Load,       UnLoad
      parameter         (Load   = 0, UnLoad = 1)
      integer            StdIn
      parameter         (StdIn  = 2)
      integer            HUnset
      parameter         (HUnset =-1)
      integer            PrintO
      parameter         (PrintO = 1 ) 
      parameter         (iPrint    =  12) ! Print file
      parameter         (iSumm     =  13) ! Summary file
      parameter         (tolfac    =  66) ! LU factor tolerance.
      parameter         (tolupd    =  67) ! LU update tolerance.
      parameter         (lvlSrt    =  69) ! = 0(1) => cold(warm) start
      parameter         (lvlHes    =  72) ! 0,1,2  => LM, FM, SD Hessian
      parameter         (lprSol    =  84) ! > 0    => print the solution 
      parameter         (Htype     = 202) ! Current Hessian type
      parameter         (nName     = 233) ! # of row and col. names
      parameter         (mProb     =  51) ! Problem name 
      character*8        cdummy
      parameter         (cdummy    = '-1111111')
      double precision   zero
      parameter         (zero      = 0.0d+0)
      parameter         (lencw     = 500)
      character*8     cw(lencw)
*     ------------------------------------------------------------------
      iError = 0

*     ------------------------------------------------------------------
*     Check memory limits and fetch the workspace starting positions.
*     ------------------------------------------------------------------
      call s2Mem ( iError, lencw, leniw, lenrw, iw,
     &     mincw, miniw, minrw, maxcw, maxiw, maxrw )

      if (iError .ne. 0) then
         inform = iError
         go to 999
      end if

*     Initialize timers and the standard input file.

      call s1time( 0, 0, iw, leniw, rw, lenrw  )
      call s1file( StdIn, iw, leniw )

      iError  = 0

*     Load the local problem dimensions.

      nCon       = nclin + ncnln

      if (nCon .eq. 0) then

*        The problem is unconstrained.
*        Include a dummy row of zeros.

         nnCol = 0
         m     = 1
         ne    = 1
      else
         nnCol = n
         m     = nCon
         ne    = m*n 
      end if

      nb         = n     + m
      nlocJ      = n     + 1

*     Load the iw array with various problem dimensions.
*     First record problem dimensions for smart users to access in iw.

      iObj    = 0
      neG     = ncnln*n
      nnCon   = ncnln
      nnJac   = nnCol
      nnObj   = n
      nGobj   = nnObj
     
      iw( 15) = n     ! copy of the number of columns
      iw( 16) = m     ! copy of the number of rows
      iw( 17) = ne    ! copy of the number of nonzeros in Jcol
      iw( 20) = neG   ! # of nonzero elems in J
      iw( 21) = nnJac ! # nonlinear Jacobian variables
      iw( 22) = nnObj ! # objective variables (usually nGobj)
      iw( 23) = nnCon ! # of nonlinear constraints
      iw( 26) = nGobj ! length of Gobj
      iw(204) = iObj  ! position of the objective row in J

      iw(nName)  = 1

      nnCon0     = max ( nnCon, 1 )

*     ------------------------------------------------------------------
*     The obligatory call to npInit has already ``unset'' 
*     the optional parameters.  However, it could not undefine 
*     the char*8 options.  Do it now.
*     Check that the optional parameters have sensible values.
*     Print the options if iPrint > 0, Print level > 0 and lvlPrm > 0.
*     ------------------------------------------------------------------
      do i = 51, 180
         cw(i)  = cdummy
      end do

*     Set default options that relate specially to npopt.

      if (rw(tolfac) .lt. zero) rw(tolfac) = 1.1d+0
      if (rw(tolupd) .lt. zero) rw(tolupd) = 1.1d+0
      if (iw(lvlHes) .lt. 0   ) iw(lvlHes) = FM

      ObjAdd   = zero

*     Check that the optional parameters have sensible values.

      call s8dflt( PrintO, cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Determine storage requirements using the
*     following variables:
*         m,      n,    ne
*         lenR ,  maxS, nnL
*         nnObj,
*         neG, nnCon, nnJac
*     All have to be known before calling s8Mem.
*     ------------------------------------------------------------------

*     Allocate arrays that are arguments of s8solv.
*     These are for the data,
*              locJ, indJ, Jcol, bl, bu, Names,
*     and for the solution
*              hs, x, pi, rc, hs.

      lNames     = mincw  - 1   ! No names

      lindJ      = miniw
      llocJ      = lindJ  + ne
      lhs        = llocJ  + nlocJ
      miniw      = lhs    + nb

      lJcol      = minrw
      lbl        = lJcol  + ne
      lbu        = lbl    + nb
      lx         = lbu    + nb
      lpi        = lx     + nb
      lrc        = lpi    + m
      minrw      = lrc    + nb

      maxR    = iw( 52) ! max columns of R.
      maxS    = iw( 53) ! max # of superbasics
      mQNmod  = iw( 54) ! (ge 0) max # of BFGS updates

      lenR    = maxR*(maxR + 1)/2

      call s8Mem ( iError, iw(iPrint), iw(iSumm), 
     &     m, n, ne, neG, nnCon, nnJac, nnObj,
     &     lenR, maxS, mQNmod, iw(lvlHes), 
     &     maxcw, maxiw, maxrw, lencw, leniw, lenrw,
     &     mincw, miniw, minrw, iw )
      if (iError .ne. 0) then
         inform = iError
         go to 999
      end if

      iw( 28)   = lenR    ! R(lenR) is the reduced Hessian factor

      iw(256)   = lJcol   ! Jcol(neJ)   = Constraint Jacobian by columns   
      iw(257)   = llocJ   ! locJ(n+1)   = column pointers for indJ  
      iw(258)   = lindJ   ! indJ(ne) holds the row indices for Jij   
      iw(271)   = lbl     ! bl(nb)      = lower bounds 
      iw(272)   = lbu     ! bu(nb)      = upper bounds 
      iw(299)   = lx      ! x(nb)       = the solution (x,s) 
      iw(279)   = lpi     ! pi(m)       = the pi-vector 
      iw(280)   = lrc     ! rc(nb)      = the reduced costs 
      iw(282)   = lhs     ! the column state vector
      iw(353)   = lNames  ! Names(nName)

      lGobj     = iw(296) ! Gobj(nnObj) = Objective gradient
      lFcon     = iw(316) ! Fcon (nnCon) constraints at x
      lGcon     = iw(320) ! Gcon (neG)   constraint gradients at x

*     ------------------------------------------------------------------
*     Load a generic problem name.
*     ------------------------------------------------------------------
      cw(mProb) = '     NLP'

      if (iw(lvlSrt) .eq. 0) then
         Start = Cold
      else
         Start = Warm
      end if

*     ------------------------------------------------------------------
*     Load the SNOPT arrays.
*     ------------------------------------------------------------------
      call npload( Load, Start,
     &     ldA, ldcJ, ldH, m, n, ncLin, nCon, nnCol,
     &     nb, nnCon0, nnCon,
     &     iw(lhs), iState,
     &     A, ne, nlocJ, iw(llocJ), iw(lindJ), rw(lJcol),
     &     bl, bu, rw(lbl), rw(lbu), cCon, cJac, cMul,
     &     rw(lFcon), rw(lGcon), rw(lGobj), grad,
     &     Hess, rw(lpi), rw(lrc), x, rw(lx),
     &     iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Solve the problem.
*     Tell s8solv that we don't have an initial Hessian.
*     ------------------------------------------------------------------
      iw(Htype)  = HUnset
      lprSav     = iw(lprSol)
      iw(lprSol) = 0

      call s8solv( Cold, npwrap, funcon, funobj, snprnt, sqprnt,
     &     m, n, nb, iw(nName), iObj, ObjAdd, Fobj, ObjTru, nInf, sInf,
     &     ne, nlocJ, iw(llocJ), iw(lindJ), rw(lJcol),
     &     rw(lbl), rw(lbu),cw(lNames),
     &     iw(lhs), rw(lx), rw(lpi), rw(lrc), inform, nMajor, nS,
     &     cw, lencw, iw, leniw, rw, lenrw,
     &     cw, lencw, iw, leniw, rw, lenrw )
      iw(lprSol) = lprSav

      Objf   = Fobj
      MajIts = nMajor

*     ------------------------------------------------------------------
*     Unload the SNOPT arrays.
*     ------------------------------------------------------------------
      call npload( Unload, Cold,
     &     ldA, ldcJ, ldH, m, n, ncLin, nCon, nnCol,
     &     nb, nnCon0, nnCon,
     &     iw(lhs), iState,
     &     A, ne, nlocJ, iw(llocJ), iw(lindJ), rw(lJcol),
     &     bl, bu, rw(lbl), rw(lbu), cCon, cJac, cMul,
     &     rw(lFcon), rw(lGcon), rw(lGobj), grad, 
     &     Hess, rw(lpi), rw(lrc), x, rw(lx),
     &     iw, leniw, rw, lenrw )

      xNorm  = dnrm1s( n, rw(lx), 1 )

      call npprnt( n, (n+nCon), ncLin, nnCon0, ldA, iw(lprSol), xNorm,
     &     iState, A, bl, bu, cCon, cMul, x, rw(lx),
     &     iw, leniw, rw, lenrw )

*     Print times for all clocks (if lvlTim > 0).

      call s1time( 0, 2, iw, leniw, rw, lenrw )

  999 return

 9000 format(  ' EXIT -- NPOPT integer and real work arrays',
     &         ' must each have at least 500 elements')

      end ! of  npopt

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine npwrap( modefg, iError, Status, getCon, getObj,
     &     n, neG, nnL, nnCon0, nnCon, nnJac, nnObj, 
     &     fgcon, fgobj,
     &     ne, nlocJ, locJ, indJ, 
     &     Fcon, Fobj, Gcon, Gobj, x, 
     &     cu, lencu, iu, leniu, ru, lenru, 
     &     cw, lencw, iw, leniw, rw, lenrw )

      implicit
     &     none
      external
     &     fgcon, fgobj
      logical
     &     getCon, getObj
      integer
     &     iError, lencu, leniu, lenru, lencw, leniw, lenrw, modefg,
     &     n, ne, neG, nlocJ, nnL, nnCon0, nnCon, nnJac, nnObj, Status,
     &     indJ(ne), locJ(nlocJ), iu(leniu), iw(leniw)
      double precision
     &     Fobj, Fcon(nnCon0),  Gobj(nnL), Gcon(neG), x(nnL),
     &     ru(lenru), rw(lenrw)
      character*8
     &     cu(lencu), cw(lencw)

*     ==================================================================
*     npwrap  calls the user-written routines  fgcon  and  fgobj  to
*     evaluate the problem functions and possibly their gradients.
*
*     Arguments  fgcon  and  fgobj  are called using modefg to control
*     the gradients as follows:
*
*     modefg        Task
*     ------        ----
*       2     Assign Fcon, Fobj and all known elements of Gcon and Gobj.
*       1     Assign all known elements of Gcon and Gobj.
*             (Fobj and Fcon are ignored).
*       0     Assign Fobj, Fcon.  (Gcon and Gobj are ignored).
*
*     09-Jan 1992: First version based on snwrap.
*     11 Nov 2000: Current version.
*     ==================================================================
      logical
     &     FPonly, gotgu, scaled
      integer
     &     Gotg2, iPrint, iSumm, lGconu, liy1, lvlScl, lvlTim, lvlDer,
     &     lAscal, lx0, lxscal, minmax, modeC, modeF, nFcon1, nFcon2,
     &     nFobj1, nFobj2
      double precision
     &     ddot
*     ------------------------------------------------------------------
      parameter         (nFcon1    = 189) ! calls to Fcon: mode = 0
      parameter         (nFcon2    = 190) ! calls to Fcon  mode > 0
      parameter         (nFobj1    = 194) ! calls to Fobj: mode = 0
      parameter         (nFobj2    = 195) ! calls to Fobj: mode > 0

      double precision   half,            one
      parameter         (half   = 0.5d+0, one   = 1.0d+0)
*     ------------------------------------------------------------------
      iPrint    = iw( 12) ! Print file
      iSumm     = iw( 13) ! Summary file
      lvlDer    = iw( 70) ! = 0, 1 or 2, the derivative level
      lvlScl    = iw( 75) ! scale option
      lvlTim    = iw( 77) ! Timing level
      minmax    = iw( 87) ! 1, 0, -1  => MIN, FP, MAX
      Gotg2     = iw(187) ! number of Gcon elements set

      lAscal    = iw(295) ! Ascale(nb) = row and column scales
      lx0       = iw(298) ! x0(nnL)    = Feasible starting point
      lxscal    = iw(302) ! xscl(n)    = copy of scaled x
      liy1      = iw(309) ! iy1(nb)    =  integer work vector
      lGconu    = iw(319) ! record of unknown derivatives and constants

      iError     = 0

      FPonly     = minmax .eq. 0
      scaled     = lvlScl .eq. 2

      modeC      = modefg
      modeF      = modefg

*     ------------------------------------------------------------------
*     Unscale x.
*     ------------------------------------------------------------------
      if ( scaled ) then
         call dcopy ( nnL, x         , 1, rw(lxscal), 1 )
         call ddscl ( nnL, rw(lAscal), 1, x         , 1 )

*        If the Jacobian is known and scaled, any constant elements
*        saved in Gconu must be copied into Gcon.

         if ( getCon ) then
            gotgU = Status .ne. 1  .and.
     &              lvlDer .ge. 2  .and.  Gotg2 .lt. neG
            if (gotgU  .and.  modefg .gt. 0) then
               call dcopy ( neG, rw(lGconu), 1, Gcon, 1 )
            end if
         end if
      end if

*     ------------------------------------------------------------------
*     Compute the constraints.
*     ------------------------------------------------------------------
*     To incorporate user workspace in fgcon, replace the next
*     call to fgcon with:
*     call fgcon ( modeC, nnCon, nnJac, nnCon,
*    &             iw(liy1), x, Fcon, Gcon, Status,
*    &             cu, lencu, iu, leniu, ru, lenru )

      if ( getCon ) then
         if (lvlTim .ge. 2) call s1time( 4, 0, iw, leniw, rw, lenrw )
         call iload ( nnCon, (1), iw(liy1), 1 )
         call fgcon ( modeC, nnCon, nnJac, nnCon,
     &        iw(liy1), x, Fcon, Gcon, Status )
         if (lvltim .ge. 2) call s1time(-4, 0, iw, leniw, rw, lenrw )

         iw(nFcon1) = iw(nFcon1) + 1
         if (modefg .gt. 0)
     &        iw(nFcon2) = iw(nFcon2) + 1
      end if

*     ------------------------------------------------------------------
*     Compute the objective.
*     ------------------------------------------------------------------
*     To incorporate user workspace in fgobj, replace the next
*     call to fgobj with:
*     call fgobj ( modeF, nnObj, x, Fobj, Gobj, Status,
*    &             cu, lencu, iu, leniu, ru, lenru )

      if (getObj  .and.  modeC .ge. 0) then

         if ( FPonly ) then
            call dcopy ( nnL, x, 1, Gobj, 1 )
            call daxpy ( nnL, (-one), rw(lx0), 1, Gobj, 1 )
            Fobj = half*ddot ( nnL, Gobj, 1, Gobj, 1 )
         else
            if (lvlTim .ge. 2) call s1time( 5, 0, iw, leniw, rw, lenrw )
            call fgobj ( modeF, nnObj, x, Fobj, Gobj, Status )
            if (lvlTim .ge. 2) call s1time(-5, 0, iw, leniw, rw, lenrw )
            iw(nFobj1) = iw(nFobj1) + 1
            if (modefg .gt. 0)
     &           iw(nFobj2) = iw(nFobj2) + 1
         end if
      end if

*     ------------------------------------------------------------------
*     Scale  x and the derivatives.
*     ------------------------------------------------------------------
      if ( scaled ) then
         call dcopy ( nnObj, rw(lxscal), 1, x, 1 )

         if ( getCon ) then
            call dddiv ( nnCon, rw(lAscal+n), 1, Fcon, 1 )
            if (modefg .gt. 0) then
               call s8sclJ( nnCon, nnJac, neG, n, rw(lAscal), 
     &              ne, nlocJ, locJ, indJ, Gcon,
     &              iw, leniw, rw, lenrw )
            end if
         end if
         
         if (getObj  .and.  modeC .ge. 0) then
            if (modefg .gt. 0) then
               call s8sclg( nnObj, rw(lAscal), Gobj, 
     &              iw, leniw, rw, lenrw )
            end if
         end if
      end if

      if (modeC .lt. 0  .or.  modeF .lt. 0) then
*        ---------------------------------------------------------------
*        The user may be saying the function is undefined (mode = -1)
*        or may just want to stop                         (mode < -1).
*        ---------------------------------------------------------------
         if (modeC .eq. -1  .or.  modeF .eq. -1) then
            iError = -1
         else
            iError =  6
            if (modeC .lt. 0) then
               if (iPrint .gt. 0) write(iPrint, 9061) iw(nFcon1)
               if (iSumm  .gt. 0) write(iSumm , 9061) iw(nFcon1)
            else
               if (iPrint .gt. 0) write(iPrint, 9062) iw(nFobj1)
               if (iSumm  .gt. 0) write(iSumm , 9062) iw(nFobj1)
            end if
         end if
      end if

      return

 9061 format(  ' EXIT -- Termination requested by User in',
     &         ' constraint subroutine after', i8, '  calls')
 9062 format(  ' EXIT -- Termination requested by User in',
     &         ' objective subroutine after', i8, '  calls')

      end ! of npwrap

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine npHess( task, ldH, lenH, n, H, Hess,
     &     iw, leniw, rw, lenrw )

      implicit
     &     none
      integer
     &     ldH, lenH, n, task, leniw, lenrw, iw(leniw)
      double precision
     &     H(lenH), Hess(ldH,*), rw(lenrw)

*     ==================================================================
*     npHess loads the problem into SNOPT format.
*
*     07 Jul 1998: First version of npHess.
*     04 Nov 2000: Current version.
*     ==================================================================
      integer
     &     i, j, l
*     ------------------------------------------------------------------
      integer            Load,       UnLoad
      parameter         (Load   = 0, UnLoad = 1)
*     ------------------------------------------------------------------

      if (task .eq. Load) then
*        ---------------------------------------------------------------
*        Load the user-supplied Hessian Hess into H.
*        ---------------------------------------------------------------
         l = 0
         do i = 1, n
            do j = i, n
               l = l + 1
               H(l) = Hess(i,j)
            end do
         end do

      else if (task .eq. UnLoad) then
*        ---------------------------------------------------------------
*        Down load the SNOPT approximate Hessian into Hess.
*        ---------------------------------------------------------------
         l = 0
         do i = 1, n
            do j = i, n
               l = l + 1
               Hess(i,j) = H(l)
               Hess(j,i) = H(l)
            end do
         end do
      end if

      end ! of  npHess

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine npload( task, Start,
     &     ldA, ldcJ, ldH, m, n, ncLin, nCon, nnCol, 
     &     nb, nnCon0, nnCon,
     &     hs, iState,
     &     Alin, ne, nlocJ, locJ, indJ, Jcol,
     &     bl, bu, bbl, bbu, c, cJac, cMul,
     &     Fcon, Gcon, Gobj, grad,
     &     Hess, pi, rc, x, xs,
     &     iw, leniw, rw, lenrw )

      implicit
     &     none
      integer
     &     ldA, ldcJ, ldH, m, n, ncLin, nCon, nnCol, Start, task, nb,
     &     nnCon0, nnCon, ne, nlocJ, leniw, lenrw, indJ(ne), hs(nb),
     &     iState(n+nCon), locJ(nlocJ), iw(leniw)
      double precision
     &     Alin(ldA,*), bl(n+nCon), bu(n+nCon), bbl(nb), bbu(nb),
     &     c(nnCon0), cJac(ldcJ,*), cMul(n+nCon), Fcon(nnCon0),
     &     Gcon(nnCon0,*), grad(n), Gobj(n), Hess(ldH,*), Jcol(ne),
     &     pi(nnCon0), rc(nb), x(n), xs(nb), rw(lenrw)

*     ==================================================================
*     npload loads a problem in NPSOL format into SNOPT format.
*
*     22 Mar 1997: First version of npload.
*     04 Nov 2000: Current version.
*     ==================================================================
      integer
     &     i, iJ, is, j, js, l, lenfH, lfH, lH0, lvlHes 
      double precision
     &     plInfy, xj
*     ------------------------------------------------------------------
      integer            Load,       UnLoad
      parameter         (Load   = 0, UnLoad = 1)
      integer            Cold,       Warm 
      parameter         (Cold   = 0, Warm   = 2)
      integer            LM   ,      FM
      parameter         (LM     = 0, FM     = 1) 
      double precision   zero
      parameter         (zero   = 0.0d+0)
*     ------------------------------------------------------------------
      plInfy    = rw( 70) ! definition of plus infinity.
      lvlHes    = iw( 72) ! 0,1,2  => LM, FM, Exact Hessian
      lH0       = iw(346) ! Initial diagonal Hessian

      if (task .eq. Load) then
*        ===============================================================
*        Load the snopt arrays.
*        Copy the bounds, x's first, linears next, then nonlinears.
*        ===============================================================
         call dcopy ( n, bl, 1, bbl, 1 ) 
         call dcopy ( n, bu, 1, bbu, 1 ) 

         if (ncLin .gt. 0) then
            call dcopy ( ncLin, bl(n+1), 1, bbl(n+nnCon+1), 1 ) 
            call dcopy ( ncLin, bu(n+1), 1, bbu(n+nnCon+1), 1 ) 
         end if

         if (nnCon .gt. 0) then
            call dcopy ( nnCon, bl(n+ncLin+1), 1, bbl(n+1), 1 ) 
            call dcopy ( nnCon, bu(n+ncLin+1), 1, bbu(n+1), 1 ) 
         end if

         if (nCon .eq. 0) then
            bbl(nb) = - plInfy  
            bbu(nb) =   plInfy 
         end if

         if (Start .eq. Cold) then
*           --------------------------------------------
*           Cold Start.
*           --------------------------------------------
            do  j = 1, n
               xj = x(j)
               if (xj .le. bl(j)) then
                  hs(j) = 4
               else if (xj .ge. bu(j)) then
                  hs(j) = 5
               else
                  hs(j) = 0
               end if
               xs(j) = xj
            end do

            if (nnCon .gt. 0) then
               call dload ( nnCon, (zero), pi, 1 )
            end if

         else if (Start .eq. Warm) then
*           ----------------------------------------------
*           Warm Start.
*           Input values of x, cMul, hs and Hess are used.
*           Note: the use of Hess is unique to npopt.
*           ----------------------------------------------
            call dcopy ( n, x, 1, xs, 1 )

            if (nnCon .gt. 0) then
               call dcopy ( nnCon, c              , 1, xs(n+1), 1 )
               call dcopy ( nnCon, cMul(n+ncLin+1), 1, pi     , 1 )
            end if

            if (ncLin .gt. 0) then
               call dload ( ncLin, (zero), xs(n+nnCon+1), 1 )
               do j = 1, n
                  call daxpy( ncLin, xs(j), Alin(1,j)    , 1,
     &                                      xs(n+nnCon+1), 1 )
               end do
            end if

            l = 1
            do  j = 1, n+nCon
               is = istate(j)
               js = 0
               if (is .eq. 1) then
                  js = 0
               else if (is .eq. 2) then
                  js = 1
               end if

               hs(l) = js

               if (j .eq. n  .and.  ncLin .gt. 0) then
                  l = n + nnCon
               else if (j .eq. n+ncLin) then
                  l = n + 1
               else
                  l = l + 1
               end if
            end do

            if (lvlHes .eq. LM) then
               call dcopy ( n, Hess(1,1), (ldH+1), rw(lH0), 1 )

            else if (lvlHes .eq. FM) then
               lfH   = iw(391) ! H(lenfH), full-memory BFGS Hessian
               lenfH = iw(392) !
               call npHess( task, ldH, lenfH, n, rw(lfH), Hess,
     &              iw, leniw, rw, lenrw )
            end if
         end if ! cold start

*        ---------------------------------------------------------------
*        Load the linear part of A with the linear constraints.
*        ---------------------------------------------------------------
         if (nnCol .eq. 0) then

*           Sparse dummy row

            Jcol(1) = zero
            indJ(1) = 1
            locJ(1) = 1

            do j = 2, n+1
               locJ(j) = 2
            end do

         else 
            iJ      = 1
            locJ(1) = 1
            do j = 1, n
               do i = 1, m
                  indJ(iJ) = i
                  if (i .le. nnCon) then
                     Jcol(iJ) = zero
                  else if (i .le. nCon) then
                     Jcol(iJ) = Alin(i-nnCon,j)
                  end if
                  iJ = iJ + 1
               end do
               locJ(j+1) = iJ 
            end do
         end if

      else if (task .eq. UNLOAD) then
*        ===============================================================
*        Unload the SNOPT solution into the npopt arrays
*        Copy Gcon, Gobj into cJac and grad,
*        ===============================================================
         l = 1
         do j = 1, n+nCon
            js = hs(l)
            is = 0
            if (js .eq. 0) then
               is = 1
            else if (js .eq. 1) then
               is = 2
            end if

            istate(j) = is

            if (j  .eq. n  .and.  ncLin .gt. 0) then
               l = n + nnCon
            else if (j .eq. n+ncLin) then
               l = n + 1
            else
               l = l + 1
            end if
         end do

*        ---------------------------------------------------------------
*        Copy Gcon, Gobj into cJac and grad,
*        ---------------------------------------------------------------
         call dcopy ( n, xs  , 1, x   , 1 )
         call dcopy ( n, Gobj, 1, grad, 1 )
         call dcopy ( n, rc  , 1, cMul, 1 ) 
         if (ncLin .gt. 0)
     &   call dcopy ( ncLin, rc(n+nnCon+1), 1, cMul(n+1), 1 )

         if (nnCon .gt. 0) then
            call dcopy ( nnCon, Fcon   , 1, c              , 1 )
            call dcopy ( nnCon, rc(n+1), 1, cMul(n+ncLin+1), 1 )

            do j = 1, n
               do i = 1, nnCon
                  cJac(i,j) = Gcon(i,j)
               end do
            end do
         end if

         if (lvlHes .eq. FM) then
            lfH   = iw(391) ! H(lenfH), full-memory BFGS Hessian
            lenfH = iw(392) !
            call npHess( task, ldH, lenfH, n, rw(lfH), Hess,
     &           iw, leniw, rw, lenrw )
         end if
      end if

      end ! of  npload

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine npprnt( n, nb, ncLin, nnCon0, ldA, lprSol, xNorm,
     &     iState, A, bl, bu, c, cMul, x, r,
     &     iw, leniw, rw, lenrw )

      implicit
     &     none
      integer
     &     n, nb, ncLin, nnCon0, ldA, lprSol, leniw, lenrw,
     &     iState(nb), iw(leniw)
      double precision
     &     xNorm, A(ldA,*), c(nnCon0),
     &     bl(nb), bu(nb), cMul(nb), r(nb), x(n),
     &     rw(lenrw)

*     ==================================================================
*     npprnt  prints  x,  A*x, c(x), the bounds, the
*     multipliers, and the slacks (distance to the nearer bound).
*
*     22 Mar 1997: First version of npprnt.
*     04 Nov 2000: Current version.
*     ==================================================================
      integer
     &     iPrint, i, is, j, nplin, number
      double precision
     &     b1, b2, bigbnd, ddot, plInfy, rj, slk, slk1, slk2, tol,
     &     tolx, wlam
      character*1
     &     key
      character*2
     &     lstate(-2:4), state
      character*8
     &     name
      character*102
     &     line
      data
     &     lstate(-2) / '--' /, lstate(-1) / '++' /
      data
     &     lstate( 0) / 'FR' /, lstate( 1) / 'LL' /
      data
     &     lstate( 2) / 'UL' /, lstate( 3) / 'EQ' /
      data
     &     lstate( 4) / 'TF' /

*     ------------------------------------------------------------------
      double precision   zero
      parameter         (zero  = 0.0d+0)
*     ------------------------------------------------------------------
      iPrint = iw( 12) ! Print file
      tolx   = rw( 56) ! Minor feasibility tolerance.
      plInfy = rw( 70) ! definition of plus infinity.

      if (iPrint .eq. 0  .or.  lprSol .eq. 0) return

      nplin  = n     + ncLin
      bigbnd = plInfy
      tol    = tolx

      write(iPrint, 1000) 'Variable       '
      name   = 'variable'
      nplin  = n + ncLin

      do j = 1, nb
         b1   = bl(j)
         b2   = bu(j)
         wlam = cMul(j)

         if (j .le. n) then
            rj  = x(j)
            tol = tolx
         else 
            tol = tolx*xNorm

            if (j .le. nplin) then
               i  = j - n
               rj = ddot  ( n, A(i,1), ldA, x, 1 )
            else
               i  = j - nplin
               rj = c(i)
            end if
         end if

         slk1 = rj - b1
         slk2 = b2 - rj
         r(j) = rj

*        Reset istate if necessary.

         is   = istate(j)
         if (                  slk1 .lt. -tol) is = - 2
         if (                  slk2 .lt. -tol) is = - 1
         if (is .eq. 1  .and.  slk1 .gt.  tol) is =   0
         if (is .eq. 2  .and.  slk2 .gt.  tol) is =   0
         istate(j) = is
         state     = lstate(is)


         if (j .le. n) then
            number = j
         else if (j .le. nplin) then
            number = j - n
            if (number .eq. 1) then
               write(iPrint, 1000) 'Linear constrnt'
               name = 'lincon  '
            end if
         else
            number = j - nplin
            if (number .eq. 1) then
               write(iPrint, 1000) 'Nonlin constrnt'
               name = 'nlncon  '
            end if
         end if

*        ------------------------------------------------
*        Print a line for the jth variable or constraint.
*        ------------------------------------------------
         if (abs(slk1) .lt. abs(slk2)) then
            slk = slk1
            if (b1 .le. - bigbnd) slk = slk2 
         else
            slk = slk2
            if (b2 .ge.   bigbnd) slk = slk1 
         end if

*        Flag infeasibilities, primal and dual degeneracies, 
*        and active QP constraints that are loose in NP.
*      
         key    = ' ' 
         if (slk1 .lt. -tol  .or.       slk2  .lt. -tol) key = 'I'
         if (is   .eq.  0    .and.  abs(slk ) .le.  tol) key = 'D'
         if (is   .ge.  1    .and.  abs(wlam) .le.  tol) key = 'A'

         write(line, 2000) name, number, key, state, 
     &                     rj, b1, b2, wlam, slk

*        Reset special cases:
*           Infinite bounds
*           Zero bounds
*           Lagrange multipliers for inactive constraints
*           Lagrange multipliers for infinite bounds
*           Infinite slacks
*           Zero slacks

         if (b1  .le. - bigbnd) line(39: 54) = '      None      '
         if (b2  .ge.   bigbnd) line(55: 70) = '      None      '
         if (b1  .eq.   zero  ) line(39: 54) = '        .       '
         if (b2  .eq.   zero  ) line(55: 70) = '        .       '
         if (is  .eq.   0       .or.    
     &       wlam.eq.   zero  ) line(71: 86) = '        .       '
         if (b1  .le. - bigbnd  .and. 
     &       b2  .ge.   bigbnd) then
                                line(71: 86) = '                '
                                line(87:102) = '                '
         end if
         if (slk .eq.   zero  ) line(87:102) = '        .       '

         write(iPrint, '(a)') line
      end do

      return

 1000 format(//  1x,  a15, 2x, 'State', 6x, 'Value',
     &           7x, 'Lower bound', 5x, 'Upper bound',
     &           3x, 'Lagr multiplier', 4x, '   Slack' / )
 2000 format( 1x, a8, i6, 3x, a1, 1x, a2, 4g16.7, g16.4 )

      end ! of npprnt

