*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
*     File  snoptm.f
*
*     SNOPT with merged user objective and constraint calls,
*
*     snoptm   snwrpm
*
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine snoptm( Start, m, n, ne, nName,
     &     nnCon, nnObj, nnJac,
     &     iObj, ObjAdd, Prob,
     &     userfg,
     &     Jcol, indJ, locJ, bl, bu, Names,
     &     hs, x, pi, rc, 
     &     inform, mincw, miniw, minrw,
     &     nS, nInf, sInf, Obj,
     &     cu, lencu, iu, leniu, ru, lenru, 
     &     cw, lencw, iw, leniw, rw, lenrw )

      implicit
     &     none
      external
     &     userfg
      integer
     &     inform, iObj, lencu, leniu, lenru, lencw, leniw, lenrw, 
     &     mincw, miniw, minrw, m, n, ne, nName, nS, nInf,  nnCon,
     &     nnObj, nnJac, indJ(ne), hs(n+m), locJ(n+1), iu(leniu),
     &     iw(leniw)
      double precision
     &     sInf, Obj, ObjAdd, Jcol(ne), bl(n+m), bu(n+m), x(n+m), pi(m),
     &     rc(n+m), ru(lenru), rw(lenrw)
      character*(*)
     &     Start
      character*8
     &     Prob, Names(nName), cu(lencu), cw(lencw)

*     ------------------------------------------------------------------
*     snoptm  is a Fortran subroutine for constrained nonlinear
*     optimization.  The constraints take the form
*
*                            (   x  )
*                      bl <= (      ) <= bu,
*                            ( F(x) )
*
*     where bl and bu are constant lower and upper bounds.
*
*     o If all constraints are linear, F = J x for some sparse matrix J.
*
*     o If all constraints are nonlinear, F = Fcon(x) for some vector
*       Fcon of smooth functions.
*
*     o In general, there is a mixture of constraints of the form
*                      ( Fcon(x1) +  J2 x2 ),
*                      (   J3 x1  +  J4 x2 )
*       where the nonlinear variables x1 must appear first as shown.
*
*     o Fcon(x1) and (optionally) its partial derivatives J1(x) are set
*       in subroutine userfg (see below).
*
*     o The matrices J2, J3, J4 and the sparsity pattern of J1(x) are
*       entered column-wise in the arrays Jcol, indJ, locJ (below).
*
*     o Internally, the constraints are converted into the form
*
*           Fcon(x1) +  J2 x2  - s1      = 0,     bl <= ( x ) <= bu    
*             J3 x1  +  J4 x2       - s2 = 0            ( s )
*
*       where s = (s1,s2)  and the components of (x,s) are the
*       variables and slacks respectively.
*
*     ------------------------------------------------------------------
*     NOTE: Before calling snoptm, your calling program must call:
*     call snInit( iPrint, iSumm,
*    &             cw, lencw, iw, leniw, rw, lenrw )
*     This sets the default values of the optional parameters. You can
*     also alter the default values of iPrint and iSumm before snoptm 
*     is used.  iPrint = 0, etc, is OK.
*     ------------------------------------------------------------------
*
*     ON ENTRY:
*
*     Start   specifies how a starting basis (and certain other items)
*             are to be obtained.
*             Start = 'Cold' means that Crash should be used to choose
*                      an initial basis, unless a basis file is given
*                      via Old basis, Insert or Load in the Specs file.
*             Start = 'Basis file' means the same (but is more
*                      meaningful in the latter case).
*             Start = 'Warm' means that a basis is already defined in hs
*                      (probably from an earlier call).
*
*     m       is the number of slacks (i.e., general constraints). 
*             For LP, QP or LC  problems this means the number of rows
*             in the constraint matrix J.
*             m > 0.
*
*             For problems with no general constraints, set m = 1 and
*             impose the constraint that the sum of the variables 
*             must lie between plus and minus infinity. This gives
*             J one ``free row'' that will not alter the solution.
*
*     n       is the number of variables, excluding slacks.
*             For LP problems, this is the number of columns in J.
*             n > 0.
*
*     ne      is the number of nonzero entries in J (including the
*             Jacobian for any nonlinear constraints).
*             ne gt 0.
*
*     nName   is the number of column and row names provided in the
*             array  Names.  If nName = 1, there are NO names.
*             Generic names will be used in the printed solution.
*             Otherwise, nName = n+m and all names must be provided.
*
*     nnCon   is the number of nonlinear constraints.
*             nnCon ge 0.
*
*             a nonzero nnCon defines the row dimension of the 
*             constraint Jacobian J1(x) defined in subroutine userfg.
*
*     nnObj   is the number of nonlinear Objective variables.
*             nnObj ge 0.
*
*     nnJac   is the number of nonlinear Jacobian variables.
*             If nnCon = 0, nnJac = 0.
*             if nnCon > 0, nnJac > 0.
*
*             a nonzero nnJac defines the column dimension of the 
*             constraint Jacobian J1(x) defined in subroutine userfg.
*
*     iObj    says which row of J is a free row containing a linear
*             objective vector  c  (iObj = 0 if none).
*             iObj = 0  or  nnCon < iObj le m.
*
*     ObjAdd  is a constant that will be added to the objective.
*             Typically ObjAdd = 0.0d+0.
*
*     Prob    is an 8-character name for the problem, used in the
*             output.  A blank name can be assigned if necessary.
*
*     Jcol(ne) is the constraint Jacobian J, stored column-wise.  Every
*             element of Jcol(*) must be assigned a value.  Elements
*             in the nonlinear part (see NOTE 2 below) can be any dummy
*             value (e.g., zero) since they are initialized by snoptm at
*             the first point that is feasible with respect to the
*             linear constraints.  The linear part of Jcol(*) must
*             contain the constant Jacobian elements. 
*            
*     indJ(ne)  is the list of row indices for each nonzero in Jcol(*).
*
*     locJ(n+1) is a set of pointers to the beginning of each column of
*             the constraint matrix within Jcol(*) and indJ(*).
*             Must have locJ(1) = 1 and locJ(n+1) = ne+1.
*
*  NOTES:  1. If the problem has a nonlinear objective,
*             the first nnObj columns of Jcol and indJ belong to the
*             nonlinear objective variables.
*             Subroutine userfg deals with these variables.
*          
*          2. If the problem has nonlinear constraints,
*             the first nnJac columns of Jcol and indJ belong to the
*             nonlinear Jacobian variables, and
*             the first nnCon rows of Jcol and indJ belong to the
*             nonlinear constraints.
*             Subroutine userfg deals with these variables and
*             constraints.
*          
*          3. If nnObj > 0 and nnJac > 0, the two sets of
*             nonlinear variables overlap.  The total number of
*             nonlinear variables is nnL = max( nnObj, nnJac ).
*          
*          4. The Jacobian forms the top left corner of a and indJ.
*             If a Jacobian column j (1 le j le nnJac) contains
*             any entries Jcol(k), indJ(k) associated with nonlinear
*             constraints (1 le indJ(k) le nnCon), those entries must
*             come before any other (linear) entries.
*          
*          5. The row indices indJ(k) for a column may be in any order
*             (subject to Jacobian entries appearing first).
*             Subroutine userfg must define Jacobian entries in the
*             same order.
*          
*     bl(n+m) is the lower bounds on each variable (x,s).
*
*     bu(n+m) is the upper bounds on each variable (x,s).
*
*     Names(nName) is an character*8 array.
*             If nName =  1, Names is not used.  The printed solution
*             will use generic names for the columns and row.
*             If nName = n+m, Names(j) should contain an 8 character
*             name of the jth variable (j = 1, n+m).
*             If j = n+i, the jth variable is the ith row.
*
*     hs(n+m) sometimes contains a set of initial states for each
*             variable (x, s).  See the following NOTES.
*
*     x(n+m)  is a set of initial values for each variable (x, s).
*
*  NOTES:  1. If Start = 'Cold' or 'Basis file' and a BASIS file
*             of some sort is to be input
*             (an OLD BASIS file, INSERT file or LOAD file),
*             hs and x need not be set at all.
*
*          2. Otherwise, hs(1:n) must be defined for a cold start.
*             If nothing special is known about the problem, or if
*             there is no wish to provide special information,
*             you may set hs(j) = 0, x(j) = 0.0d+0 for all j=1:n.
*             All variables will be eligible for the initial basis.
*        
*             Less trivially, to say that variable j will probably
*             be equal to one of its bounds,
*             set hs(j) = 4 and x(j) = bl(j)
*             or  hs(j) = 5 and x(j) = bu(j) as appropriate.
*        
*          3. For Cold starts with no basis file, a Crash procedure
*             is used to select an initial basis.  The initial basis
*             matrix will be triangular (ignoring certain small
*             entries in each column).
*             The values hs(j) = 0, 1, 2, 3, 4, 5 have the following
*             meaning:
*                
*             hs(j)    State of variable j during Crash
*        
*             0, 1, 3  Eligible for the basis.  3 is given preference.
*             2, 4, 5  Ignored.
*        
*             After Crash, hs(j) = 2 entries are made superbasic.
*             Other entries not selected for the basis are made
*             nonbasic at the value x(j) if bl(j) <= x(j) <= bu(j),
*             or at the value bl(j) or bu(j) closest to x(j).
*
*          4. For Warm starts, all of hs(1:n+m) is assumed to be
*             set to the values 0, 1, 2 or 3 from some previous call.
*        
*     pi(m)   contains an estimate of the vector of Lagrange multipliers
*             (shadow prices) for the NONLINEAR constraints.  The first
*             nnCon components must be defined.  They will be used as
*             lambda in the subproblem objective function for the first
*             major iteration.  If nothing is known about lambda,
*             set pi(i) = 0.0d+0, i = 1 to nnCon.
*
*     nS      need not be specified for Cold starts,
*             but should retain its value from a previous call
*             when a Warm start is used.
*
*
*     ON EXIT:
*
*     hs(n+m) is the final state vector:
*
*                hs(j)    State of variable j    Normal value of x(j)
*
*                  0      nonbasic               bl(j)
*                  1      nonbasic               bu(j)
*                  2      superbasic             Between bl(j) and bu(j)
*                  3      basic                  ditto
*
*             Very occasionally there may be nonbasic variables for
*             which x(j) lies strictly between its bounds.
*             If nInf = 0, basic and superbasic variables may be outside
*             their bounds by as much as the Feasibility tolerance.
*             Note that if Scale is specified, the Feasibility tolerance
*             applies to the variables of the SCALED problem. 
*             In this case, the variables of the original problem may be
*             as much as 0.1 outside their bounds, but this is unlikely
*             unless the problem is very badly scaled.
*
*     x(n+m)  contains the final variables and slacks (x, s).
*
*     pi(m)   is the vector of Lagrange multipliers (shadow prices)
*             for the general constraints.
*
*     rc(n+m) is a vector of reduced costs: rc = g - (J -I)'pi, where g
*             is the gradient of the objective if x is feasible
*             (or the gradient of the Phase-1 objective otherwise).
*             If nInf = 0, the last m entries are pi.
*
*     inform  says what happened; see the User's Guide.
*             A summary of possible values follows:
*
*             Inform   Meaning
*
*                0     Optimal solution found.
*                1     The problem is infeasible.
*                2     The problem is unbounded (or badly scaled).
*                3     Too many iterations.
*                5     The Superbasics limit is too small.
*                4     Final solution is feasible, but final optimality 
*                      could not quite be achieved,
*                6     Subroutine userfg requested termination by
*                      returning mode < 0.
*                7     Subroutine userfg seems to be giving incorrect
*                      gradients.
*                8     Subroutine userfg seems to be giving incorrect
*                      gradients.
*                9     The current point cannot be improved.
*               10     Numerical error in trying to satisfy the linear
*                      constraints (or the linearized nonlinear
*                      constraints).  The basis is very ill-conditioned.
*
*               20     Not enough storage for the basis factorization.
*               21     Error in basis package.
*               22     The basis is singular after several attempts to
*                      factorize it (and add slacks where necessary).
*
*               30     An OLD BASIS file had dimensions that did not
*                      match the current problem.
*               32     System error.  Wrong number of basic variables.
*
*               41     Not enough storage to hold SNOPT local variables.
*               42     Not enough char*8  storage to solve the problem.
*               43     Not enough integer storage to solve the problem.
*               44     Not enough real    storage to solve the problem.
*
*     mincw   says how much character storage is needed to solve the
*             problem.  If inform = 42, the work array cw(lencw) was 
*             too small.  snoptm may be called again with lencw suitably 
*             larger than mincw.
*
*     miniw   says how much integer storage is needed to solve the
*             problem.  If inform = 43, the work array iw(leniw) was too
*             small.  snoptm may be called again with leniw suitably 
*             larger than miniw.  (The bigger the better, since it is
*             not certain how much storage the basis factors need.)
*
*     minrw   says how much real storage is needed to solve the problem.
*             If inform = 44, the work array rw(lenrw) was too small.
*             (See the comments above for miniw.)
*
*     nS      is the final number of superbasics.
*
*     nInf    is the number of infeasibilities.
*
*     sInf    is the sum    of infeasibilities.
*
*     Obj     is the value of the nonlinear part of the objective.
*             If nInf = 0, Obj includes the nonlinear objective if any.
*             If nInf > 0, Obj is just the linear objective if any.
*
*     cu(lencu), iu(leniu), ru(lenru)  are character, integer and real
*             arrays of USER workspace.  These arrays are available to
*             pass data to the user-defined routine userfg.
*             If no workspace is required, you can either use dummy
*             arrays for cu, iu and ru, or use cw, iw and rw
*             (see below).
*
*     cw(lencw), iw(leniw), rw(lenrw)  are character*8, integer and real
*             arrays of workspace used by snoptm.
*             lencw  should be about at least 500.
*             leniw  should be about max( 500, 20(m+n) ) or larger.
*             lenrw  should be about max( 500, 40(m+n) ) or larger.
*
*     SNOPTM is maintained by Philip E. Gill, 
*     Dept of Mathematics, University of California, San Diego.
*     
*     LUSOL is maintained by Michael A. Saunders,
*     Systems Optimization Laboratory,
*     Dept of Management Science & Engineering, Stanford University.
*
*     31 Oct 1998: First version based on snopt in SNOPT 5.3-4.
*     11 Nov 2000: Current version of snoptm.
*     ==================================================================
      character*1
     &     ch1
      integer
     &     Htype, iError, iPrint, iSumm, iStart, lenR, lvlHes, maxcw,
     &     maxiw, maxR, maxrw, maxS, mProb, mQNmod, nb, neG, nGobj,
     &     nlocJ, nMajor
      double precision
     &     Fobj, ObjTru
      external
     &     snprnt, sqprnt, snwrpm
*     ------------------------------------------------------------------
      integer            Cold,       Basis,      Warm 
      parameter         (Cold   = 0, Basis  = 1, Warm  = 2)
       integer            StdIn
      parameter         (StdIn  = 2)
      integer            HUnset
      parameter         (HUnset =-1)
      integer            PrintO
      parameter         (PrintO = 1 ) 
      parameter         (iPrint    =  12) ! Print file
      parameter         (iSumm     =  13) ! Summary file
      parameter         (neG       =  20) ! # of nonzero elems in J
      parameter         (Htype     = 202) ! Current Hessian type
      parameter         (mProb     =  51) ! Problem name 
*     ------------------------------------------------------------------
      iError = 0

*     ------------------------------------------------------------------
*     Check memory limits and fetch the workspace starting positions.
*     ------------------------------------------------------------------
      call s2Mem ( iError, lencw, leniw, lenrw, iw,
     &     mincw, miniw, minrw, maxcw, maxiw, maxrw )

      if (iError .ne. 0) then
         inform = iError
         go to 999
      end if

*     Initialize timers and the standard input file.

      call s1time( 0, 0, iw, leniw, rw, lenrw  )
      call s1file( StdIn, iw, leniw )

      iError  = 0

*     Load the iw array with various problem dimensions.
*     Load the iw array with various problem dimensions.

      nGobj   = nnObj

      iw( 15) = n     ! copy of the number of columns
      iw( 16) = m     ! copy of the number of rows
      iw( 17) = ne    ! copy of the number of nonzeros in Jcol
      iw( 21) = nnJac ! # nonlinear Jacobian variables
      iw( 22) = nnObj ! # objective variables (usually nGobj)
      iw( 23) = nnCon ! # of nonlinear constraints
      iw( 26) = nGobj ! length of Gobj
      iw(204) = iObj  ! position of the objective row in J

*     ------------------------------------------------------------------
*     The obligatory call to snInit has already set the defaults.
*     Check that the optional parameters have sensible values.
*     Print the options if iPrint > 0, Print level > 0 and lvlPrm > 0.
*     ------------------------------------------------------------------
      call s8dflt( PrintO, cw, lencw, iw, leniw, rw, lenrw )

*     ------------------------------------------------------------------
*     Compute the storage requirements for SNOPT  from the following
*     variables:
*         m,      n,    ne
*         lenR ,  maxS, nnL
*         nnObj,
*         neG, nnCon, nnJac
*     All have to be known exactly before calling s8Mem.
*     The only one in doubt is neG, the number of Jacobian elements.
*     Count them here.
*     ------------------------------------------------------------------
      nlocJ   = n + 1
      call s8Gsiz( m, nnCon, nnJac, ne, nlocJ, locJ, indJ, iw(neG) )

      maxR    = iw( 52) ! max columns of R.
      maxS    = iw( 53) ! max # of superbasics
      mQNmod  = iw( 54) ! (ge 0) max # of BFGS updates
      lvlHes  = iw( 72) ! 0,1,2  => LM, FM, Exact Hessian

      lenR    = maxR*(maxR + 1)/2
      iw( 28) = lenR

      call s8Mem ( iError, iw(iPrint), iw(iSumm), 
     &     m, n, ne, iw(neG), nnCon, nnJac, nnObj,
     &     lenR, maxS, mQNmod, lvlHes, 
     &     maxcw, maxiw, maxrw, lencw, leniw, lenrw,
     &     mincw, miniw, minrw, iw )
      if (iError .ne. 0) then
         inform = iError
         go to 999
      end if

*     ------------------------------------------------------------------
*     Save the problem name.
*     ------------------------------------------------------------------
      cw(mProb) = Prob

*     ==================================================================
*     Decode 'Start'.
*     ==================================================================
      ch1    = Start(1:1)

      if      (ch1 .eq. 'C'  .or.  ch1 .eq. 'c') then
         iStart = COLD
      else if (ch1 .eq. 'B'  .or.  ch1 .eq. 'b') then
         iStart = BASIS
      else if (ch1 .eq. 'W'  .or.  ch1 .eq. 'w') then
         iStart = WARM
      else
         iStart = COLD
         if (iPrint .gt. 0) write(iPrint, 1030) Start
         if (iSumm  .gt. 0) write(iSumm , 1030) Start
      end if

*     ------------------------------------------------------------------
*     Solve the problem.
*     Tell s8solv that we don't have an initial Hessian.
*     ------------------------------------------------------------------
      nb         = n + m
      iw(Htype)  = HUnset
      call s8solv( iStart, snwrpm, userfg, userfg, snprnt, sqprnt,
     &     m, n, nb, nName, iObj, ObjAdd, Fobj, ObjTru, nInf, sInf,
     &     ne, nlocJ, locJ, indJ, Jcol, bl, bu, Names, 
     &     hs, x, pi, rc, inform, nMajor, nS,
     &     cu, lencu, iu, leniu, ru, lenru, 
     &     cw, lencw, iw, leniw, rw, lenrw )

      Obj    = ObjTru

*     Print times for all clocks (if lvlTim > 0).

      call s1time( 0, 2, iw, leniw, rw, lenrw )

  999 return

 1030 format(/ ' XXX Start parameter not recognized:  ', a)
 9000 format(  ' EXIT -- snoptm character, integer and real',
     &         ' work arrays each must have at least 500 elements')

      end ! of snoptm

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine snwrpm( modefg, iError, Status, getCon, getObj,
     &     n, neG, nnL, nnCon0, nnCon, nnJac, nnObj, 
     &     userfg, dummy,
     &     ne, nlocJ, locJ, indJ, 
     &     Fcon, Fobj, Gcon, Gobj, x, 
     &     cu, lencu, iu, leniu, ru, lenru, 
     &     cw, lencw, iw, leniw, rw, lenrw )

      implicit
     &     none
      external
     &     userfg, dummy
      logical
     &     getCon, getObj
      integer
     &     iError, lencu, leniu, lenru, lencw, leniw, lenrw, modefg,
     &     n, ne, neG, nlocJ, nnL, nnCon0, nnCon, nnJac,
     &     nnObj, Status, indJ(ne), locJ(nlocJ), iu(leniu), iw(leniw)
      double precision
     &     Fobj, Fcon(nnCon0),  Gobj(nnL), Gcon(neG), x(nnL),
     &     ru(lenru), rw(lenrw)
      character*8
     &     cu(lencu), cw(lencw)

*     ==================================================================
*     snwrpm  calls the user-written routine  userfg  to evaluate the
*     problem functions and possibly their gradients.
*
*     Argument  userfg  is called using modefg to control
*     the gradients as follows:
*
*     modefg        Task
*     ------        ----
*       2     Assign Fcon, Fobj and all known elements of Gcon and Gobj.
*       1     Assign all known elements of Gcon and Gobj.
*             (Fobj and Fcon are ignored).
*       0     Assign Fobj, Fcon.  (Gcon and Gobj are ignored).
*
*     Since objective and constraints are computed simultaneously,
*     the input variables  getCon  and  getObj are ignored. 
*
*     31 Oct 1998: First version based on snwrap in SNOPT 5.3-4.
*     11 Nov 2000: Current version of snwrpm.
*     ==================================================================
      logical
     &     FPonly, gotgU, scaled
      integer
     &     Gotg2, iPrint, iSumm, lGconu, lvlScl, lvlTim, lvlDer, lAscal,
     &     lx0, lxscal, minmax, mode, nFcon1, nFcon2, nFobj1, nFobj2,
     &     nGobj
      double precision
     &     ddot
*     ------------------------------------------------------------------
      parameter         (nFcon1    = 189)
      parameter         (nFcon2    = 190)
      parameter         (nFobj1    = 194)
      parameter         (nFobj2    = 195)
      double precision   half,            one
      parameter         (half   = 0.5d+0, one   = 1.0d+0)
*     ------------------------------------------------------------------
      iPrint    = iw( 12) ! Print file
      iSumm     = iw( 13) ! Summary file
      nGobj     = iw( 26) ! # elements of  gObj
      lvlDer    = iw( 70) ! = 0, 1 or 2, the derivative level
      lvlScl    = iw( 75) ! scale option
      lvlTim    = iw( 77) ! Timing level
      minmax    = iw( 87) ! 1, 0, -1  => MIN, FP, MAX
      Gotg2     = iw(187) ! number of Gcon elements set

      lAscal    = iw(295) ! Ascale(nb)  = row and column scales
      lx0       = iw(298) ! x0(nnL)    = Feasible starting point
      lxscal    = iw(302) ! xscl(n)     = copy of scaled  x
      lGconu    = iw(319) ! record of unknown derivatives and constants

      iError     = 0
      FPonly     = minmax .eq. 0
      scaled     = lvlScl .eq. 2

      mode       = modefg

*     ------------------------------------------------------------------
*     Unscale x.
*     ------------------------------------------------------------------
      if ( scaled ) then
         call dcopy ( nnL, x         , 1, rw(lxscal), 1 )
         call ddscl ( nnL, rw(lAscal), 1, x         , 1 )

*        If the Jacobian is known, any constant elements saved in Gconu
*        must be copied into Gcon.

         gotgU = Status .ne. 1  .and.
     &           lvlDer .ge. 2  .and.  Gotg2 .lt. neG
         if (gotgU  .and.  modefg .gt. 0) then
            call dcopy ( neG, rw(lGconu), 1, Gcon, 1 )
         end if
      end if

*     ------------------------------------------------------------------
*     Compute the user-defined functions and derivatives.
*     ------------------------------------------------------------------
      if (lvlTim .ge. 2) call s1time( 4, 0, iw, leniw, rw, lenrw )
      call userfg( mode, nGobj, nnCon, nnJac, neG,
     &     x, Fobj, Gobj, Fcon, Gcon, Status,
     &     cu, lencu, iu, leniu, ru, lenru )
      if (lvltim .ge. 2) call s1time(-4, 0, iw, leniw, rw, lenrw )

      if ( FPonly ) then
         call dcopy ( nnL, x, 1, Gobj, 1 )
         call daxpy ( nnL, (-one), rw(lx0), 1, Gobj, 1 )
         Fobj = half*ddot ( nnL, Gobj, 1, Gobj, 1 )
      end if

      iw(nFcon1) = iw(nFcon1) + 1
      iw(nFobj1) = iw(nFobj1) + 1
      if (modefg .gt. 0) then
         iw(nFcon2) = iw(nFcon2) + 1
         iw(nFobj2) = iw(nFobj2) + 1
      end if

*     ------------------------------------------------------------------
*     Scale  x and the derivatives.
*     ------------------------------------------------------------------
      if ( scaled ) then
         call dcopy ( nnL  , rw(lxscal)  , 1, x   , 1 )
         call dddiv ( nnCon, rw(lAscal+n), 1, Fcon, 1 )
         if (modefg .gt. 0) then
            call s8sclg( nnObj, rw(lAscal), Gobj, 
     &           iw, leniw, rw, lenrw )
            call s8sclJ( nnCon, nnJac, neG, n, rw(lAscal), 
     &           ne, nlocJ, locJ, indJ, Gcon, iw, leniw, rw, lenrw )
         end if
      end if

      if (mode .lt. 0) then
*        ---------------------------------------------------------------
*        The user may be saying the function is undefined (mode = -1)
*        or may just want to stop                         (mode < -1).
*        ---------------------------------------------------------------
         if (mode .eq. -1) then
            iError = -1
         else
            iError =  6
            if (iPrint .gt. 0) write(iPrint, 9060) iw(nFcon1)
            if (iSumm  .gt. 0) write(iSumm , 9060) iw(nFcon1)
         end if
      end if

      return

 9060 format(  ' EXIT -- Termination requested after', i8,
     &         '  calls to the user-supplied routine.')

      end ! of snwrpm

